<?php
/**
 * easy2 Framework - PHP framework based on PDO MySQL and Ajax integration
 * PHP Version 7.4
 * @author Daniel Bernát
 */


/**
 * RKO Module
 * Handles communication with RKO system
 */

class ProcessRko
{

    /** @var ProcessMini $process */
    public $process;

    /** @var array $estateList */
    public $estateList;

    /** @var array $hhrCzEn */
    public $hhrCzEn;

    /** @var array $hhrCzOnly */
    public $hhrCzOnly;

    /** @var array $tempArray */
    public $tempArray;

    function __construct($parent)
    {
        $this->process = $parent;
    }

    public function translate($phrase)
    {
        if (is_numeric($phrase))
        {
            return $phrase;
        }
        if (!isset($this->hhrCzEn) || !is_array($this->hhrCzEn))
        {
            $this->initHhrCzEn();
        }
        if (isset($this->hhrCzEn[$phrase]))
        {
            return $this->hhrCzEn[$phrase];
        }
        else if (isset($this->hhrCzOnly[$phrase]))
        {
            return $phrase;
        }
        else
        {
            $this->addHhrCzEnPhrase($phrase);
            $this->hhrCzOnly[$phrase] = '';
            return $phrase;
        }
    }

    public function addHhrCzEnPhrase($phrase)
    {
        if (empty($phrase))
        {
            return false;
        }
        else
        {
            try
            {
                $db = $this->process->connectDb();

                $stmt = $db->prepare("insert into rko_cz_en (cz) values(:cz)");
                empty($phrase) ? $stmt->bindValue(':cz', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':cz', $phrase);
                $stmt->execute();
            }
            catch (PDOException $e)
            {
                $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
                $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
                return null;
            }
            return true;
        }
    }

    public function initHhrCzEn()
    {
        $this->hhrCzEn = [];
        $this->hhrCzOnly = [];
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select * from rko_cz_en");
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if (isset($all) && is_array($all))
        {
            foreach ($all as $row)
            {
                $cz = $row['cz'];
                $en = $row['en'];
                if (empty($en))
                {
                    $this->hhrCzOnly[$cz] = '';
                }
                else
                {
                    $this->hhrCzEn[$cz] = $en;
                }
            }
        }
        return true;
    }

    public function getEstateRow($estateId)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select * from rko_estate_list where estate_id = :estate_id");
            empty($estateId) ? $stmt->bindValue(':estate_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':estate_id', $estateId);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        return isset($all) ? $all[0] : null;
    }

    public function getParamsOrder($contentType)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select distinct content_order, param_name from rko_content_settings where content_type = :contentType and content_category = 'Params' order by content_order, param_name");
            empty($contentType) ? $stmt->bindValue(':contentType', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':contentType', $contentType);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if(isset($all) && count($all) > 0){
            foreach($all as $row){
                $finalArray[] = $row['param_name'];
            }
        }
        return $finalArray ?? [];
    }

    public function getEstateAgents()
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select distinct agent_id from rko_estate_list where agent_id is not null");
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if (isset($all) && count($all) > 0)
        {
            foreach ($all as $row)
            {
                $finalArray[$row['agent_id']] = $row['agent_id'];
            }
        }
        $db = null;
        return isset($finalArray) ? $finalArray : null;
    }

    public function getAgentRow($agentId)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select * from rko_agent_list where agent_id = :agent_id");
            empty($agentId) ? $stmt->bindValue(':agent_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':agent_id', $agentId);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return isset($all) && isset($all[0]) ? $all[0] : [];
    }

    public function getAgentList()
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select * from rko_agent_list");
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if (isset($all) && count($all) > 0)
        {
            foreach ($all as $row)
            {
                $finalArray[$row['agent_id']] = $row;
            }
        }
        $db = null;
        return isset($finalArray) ? $finalArray : null;
    }

    public function parseEstateToArray($estateId, $returnAsTable = false)
    {
        $arrayName = 'estate:' . $estateId;
        $estateRow = $this->getEstateRow($estateId);
        if (isset($estateRow))
        {
            $estateJson = $estateRow['estate_array'];
            $estateArray = json_decode($estateJson, true);
            $rkoItemId = $estateArray['GetEstateResult']['Data']['RegistrationNumber'];
            $this->process->debugMessage['estate_item_id:' . $estateId] = $rkoItemId;
            $this->process->debugMessage['estate_ajax:' . $estateId] = $estateJson;
            $this->process->debugMessage['estate_array:' . $estateId] = $estateArray;
            $this->parseArrayRecursive($arrayName, $estateArray, '', ['GetEstateResult', 'apiDocument', 'apiEnum', 'Data', 'apiOtherLanguageText', 'apiUserField'], ['UserFields*' => 'Name', 'UserFields' => 'Name']);
        }
        if ($returnAsTable)
        {
            $table = '<table>';
            $table .= '<tr><th colspan="2"><h5>Debug data</h5></th></tr>';
            $table .= '<tr><th>Variable name</th><th>Value</th></tr>';
            foreach ($this->tempArray[$arrayName] as $key => $value)
            {
                $table .= '<tr><td>' . $key . '</td><td>' . $value . '</td></tr>';
            }
            $table .= '</table>';
            return $table;
        }
        else
        {
            return $this->tempArray[$arrayName] ?? 'No data available!';
        }
    }

    public function parseEstateFilterToArray($estateId, $returnAsTable = false)
    {
        $arrayName = 'filtr:' . $estateId;
        $estateRow = $this->getEstateRow($estateId);
        if (isset($estateRow))
        {
            $estateJson = $estateRow['estate_array'];
            $estateArray = json_decode($estateJson, true);
            $this->parseArrayRecursive($arrayName, $estateArray, '', ['GetEstateResult', 'apiDocument', 'apiEnum', 'Data', 'apiOtherLanguageText', 'apiUserField'], ['UserFields*' => 'Name', 'UserFields' => 'Name']);
        }

        $filterKey = $this->tempArray[$arrayName]['TypeOfRealEstate-StringValue'];
        $filterType = $this->tempArray[$arrayName]['TypeOfCommission-StringValue'];
        $filterCategoryName = $estateRow['filter_category'];

        $filterSettingsArray = $this->getFilterSettings($filterCategoryName);


        foreach ($filterSettingsArray as $keyData)
        {
            $key = $keyData['param_name'];
            $keyType = $keyData['param_type'];
            $mask = $keyData['param_mask'];
            $mask = str_replace('*', '(.+)', $mask);
            $mask = '/^' . $mask . '$/';
            $this->process->debugMessage[] = $mask;
            $keys = preg_grep($mask, array_keys($this->tempArray[$arrayName]));
            $this->process->debugMessage[] = $keys;
            // if (!isset($keys) || count($keys) == 0)
            // {
            //     $filterArray[$key] = 'nezjištěno';
            // }
            if (isset($keys) && count($keys) > 0)
            {
                foreach ($keys as $estateKey)
                {
                    if ($keyType == 'bool')
                    {
                        $filterArray[$key][] = isset($this->tempArray[$arrayName][$estateKey]) && $this->tempArray[$arrayName][$estateKey] == 1 ? 'ano' : 'ne';
                    }
                    else if ($keyType == 'iftrue')
                    {
                        if (isset($this->tempArray[$arrayName][$estateKey]) && $this->tempArray[$arrayName][$estateKey] == 1)
                        {
                            $filterArray[$key][] = $keyData['if_value'];
                        }
                    }
                    else if ($keyType == 'ifexists')
                    {
                        if (isset($this->tempArray[$arrayName][$estateKey]))
                        {
                            $filterArray[$key][] = $keyData['if_value'];
                        }
                    }
                    else
                    {
                        if (!isset($filterArray[$key]) || in_array($this->tempArray[$arrayName][$estateKey], $filterArray[$key]) === false)
                        {
                            $filterArray[$key][] = $this->tempArray[$arrayName][$estateKey];
                        }
                    }
                }
            }
        }

        if ($returnAsTable)
        {
            $table = '<table>';
            $table .= '<tr><th colspan="2"><h5>Filter type: ' . $filterCategoryName . '</h5></th></tr>';
            $table .= '<tr><th>Filter param</th><th>Value</th></tr>';
            foreach ($filterArray as $key => $value)
            {
                if (is_array($value) && count($value) > 0)
                {
                    $valueText = implode(', ', $value);
                }
                else
                {
                    $valueText = '[empty]';
                }
                $table .= '<tr><td>' . $key . '</td><td>' . $valueText . '</td></tr>';
            }
            $table .= '</table>';
            return $table;
        }
        else
        {
            return $filterArray ?? 'No data available!';
        }
    }

    public function parseEstateFilterToArrayExtended($estateId, $returnAsTable = false)
    {
        $arrayName = 'filtr:' . $estateId;
        $estateRow = $this->getEstateRow($estateId);
        if (isset($estateRow))
        {
            $estateJson = $estateRow['estate_array'];
            $estateArray = json_decode($estateJson, true);
            $this->parseArrayRecursive($arrayName, $estateArray, '', ['GetEstateResult', 'apiDocument', 'apiEnum', 'Data', 'apiOtherLanguageText', 'apiUserField'], ['UserFields*' => 'Name', 'UserFields' => 'Name']);
        }

        $filterKey = $this->tempArray[$arrayName]['TypeOfRealEstate-StringValue'];
        $filterType = $this->tempArray[$arrayName]['TypeOfCommission-StringValue'];

        $filterCategoryName = $estateRow['filter_category'];

        $filterSettingsArray = $this->getFilterSettings($filterCategoryName);


        foreach ($filterSettingsArray as $keyData)
        {

            $filterCategory = $keyData['filter_category'];
            $filterEvaluation = $keyData['filter_evaluation'];
            $filterOrder = $keyData['filter_order'];
            $keyName = $keyData['param_name'];
            $keyType = $keyData['param_type'];
            $mask = $keyData['param_mask'];
            $mask = str_replace('*', '(.+)', $mask);
            $mask = '/^' . $mask . '$/';
            $this->process->debugMessage[] = $mask;
            $keys = preg_grep($mask, array_keys($this->tempArray[$arrayName]));
            $this->process->debugMessage[] = $keys;
            if (isset($keys) && count($keys) > 0)
            {
                foreach ($keys as $estateKey)
                {
                    if ($keyType == 'bool')
                    {
                        $key = $keyName;
                        $filterArray[$filterCategory][$key] = isset($this->tempArray[$arrayName][$estateKey]) && $this->tempArray[$arrayName][$estateKey] == 1 ? 'ano' : 'ne';
                    }
                    else if ($keyType == 'iftrue')
                    {
                        if (isset($this->tempArray[$arrayName][$estateKey]) && $this->tempArray[$arrayName][$estateKey] == 1)
                        {
                            $key = $keyData['if_value'];
                            $filterArray[$filterCategory][$keyName] = $key;
                        }
                    } 
                    else if ($keyType == 'ifvalue')
                    {
                        if (isset($this->tempArray[$arrayName][$estateKey]) && $this->tempArray[$arrayName][$estateKey] == $keyData['test_value'])
                        {
                            $key = $keyData['if_value'];
                            $filterArray[$filterCategory][$keyName][] = $key;
                        }
                    }

                    else if ($keyType == 'ifexists')
                    {
                        if (isset($this->tempArray[$arrayName][$estateKey]))
                        {
                            $key = $keyData['if_value'];
                            $filterArray[$filterCategory][$keyName] = $key;
                        }
                    }
                    else if ($keyType === 'special')
                    {
                        if (isset($this->tempArray[$arrayName][$estateKey]) && $filterCategory === 'Makléř' && $keyName === 'Makléř')
                        {
                            $agentId = $this->tempArray[$arrayName][$estateKey];
                            $agent = $this->getAgentRow($agentId);
                            $key = $agent['firstname'] . ' ' . $agent['lastname'];
                            $filterArray[$filterCategory][$keyName][] = $key;
                        }
                    }
                    else
                    {
                        $key = $filterEvaluation == 'number' || $filterEvaluation == 'text' || $filterEvaluation == 'yesorno' || $filterEvaluation == 'checkbox' ? $keyName : $keyName;
                        if (!empty($key))
                        {
                            if ($filterEvaluation == 'text')
                            {
                                $filterArray[$filterCategory][$key] = isset($filterArray[$filterCategory][$key]) ? $filterArray[$filterCategory][$key] . ', ' . $this->tempArray[$arrayName][$estateKey] : $this->tempArray[$arrayName][$estateKey];
                            }
                            else
                            {
                                $filterArray[$filterCategory][$key] = $this->tempArray[$arrayName][$estateKey];
                            }
                        }
                    }
                    $filterSettings[$filterCategory][$key] = $filterEvaluation;
                    $filterOrders[$filterCategory][$key] = $filterOrder;
                }
            }
        }

        if ($returnAsTable)
        {
            $table = '<table>';
            $table .= '<tr><th colspan="3"><h5>Filter type: ' . $filterCategoryName . '</h5></th></tr>';
            $table .= '<tr><th>Filter category</th><th>Filter param</th><th>Value</th></tr>';
            foreach ($filterArray as $category => $keyData)
            {
                foreach ($keyData as $key => $value)
                {
                    if (is_array($value) && count($value) > 0)
                    {
                        $valueText = implode(', ', $value);
                    }
                    else
                    {
                        $valueText = $value;
                    }
                    $table .= '<tr><td>' . $category . '</td><td>' . $key . '</td><td>' . $valueText . '</td></tr>';
                }
            }
            $table .= '</table>';
            return $table;
        }
        else
        {
            return isset($filterArray) ? ['filterArray' => $filterArray, 'filterSettings' => $filterSettings, 'filterOrders' => $filterOrders, 'filterType' => $filterKey . '-' . $filterType] : [];
        }
    }

    public function parseEstateContentToArray($estateId, $returnAsTable = false, $debug = false)
    {
        $arrayName = 'filtr:' . $estateId;
        $estateRow = $this->getEstateRow($estateId);
        if (isset($estateRow))
        {
            $estateJson = $estateRow['estate_array'];
            $estateArray = json_decode($estateJson, true);
            $this->parseArrayRecursive($arrayName, $estateArray, '', ['GetEstateResult', 'apiDocument', 'apiEnum', 'Data', 'apiOtherLanguageText', 'apiUserField'], ['UserFields*' => 'Name', 'UserFields' => 'Name']);
        }

        $filterKey = $this->tempArray[$arrayName]['TypeOfRealEstate-StringValue'];
        $filterType = $this->tempArray[$arrayName]['TypeOfCommission-StringValue'];

        $filterCategoryName = $estateRow['filter_category'];

        $filterSettingsArray = $this->getContentSettings($filterCategoryName);


        foreach ($filterSettingsArray as $keyData)
        {
            $category = $keyData['content_category'];
            $key = $keyData['param_name'];
            $keyType = $keyData['param_type'];
            $mask = $keyData['param_mask'];
            $debug ? $this->process->debugMessage[] = $mask : true;
            $mask = str_replace('*', '(.+)', $mask);
            $debug ? $this->process->debugMessage[] = $mask : true;
            $mask = '/^' . $mask . '$/';
            $debug ? $this->process->debugMessage[] = $mask : true;
            $keys = preg_grep($mask, array_keys($this->tempArray[$arrayName]));
            $debug ? $this->process->debugMessage[] = $keys : true;
            if (isset($keys) && count($keys) > 0)
            {
                foreach ($keys as $estateKey)
                {
                    if ($keyType == 'bool')
                    {
                        $filterArray[$category][$key][] = isset($this->tempArray[$arrayName][$estateKey]) && $this->tempArray[$arrayName][$estateKey] == 1 ? 'ano' : 'ne';
                    }
                    else if ($keyType == 'iftrue')
                    {
                        if (isset($this->tempArray[$arrayName][$estateKey]) && $this->tempArray[$arrayName][$estateKey] == 1)
                        {
                            $filterArray[$category][$key][] = $keyData['if_value'];
                        }
                    }
                    else if ($keyType == 'ifexists')
                    {
                        if (isset($this->tempArray[$arrayName][$estateKey]))
                        {
                            $filterArray[$category][$key][] = $keyData['if_value'];
                        }
                    }
                    else if ($keyType === 'special')
                    {
                        if (isset($this->tempArray[$arrayName][$estateKey]) && $category === 'Content' && $key === 'Cena / m2')
                        {
                            $priceTotal = $estateArray['GetEstateResult']['Data']['Price'];
                            $priceDivider = $estateArray['GetEstateResult']['Data']['AreaFloor'];
                            $priceFinal = round($priceTotal / $priceDivider);

                            $filterArray[$category][$key][] = number_format($priceFinal);
                        }
                    }
                    else
                    {
                        if (!isset($filterArray[$category][$key]) || in_array($this->tempArray[$arrayName][$estateKey], $filterArray[$category][$key]) === false)
                        {
                            $filterArray[$category][$key][] = $this->tempArray[$arrayName][$estateKey];
                        }
                    }
                }
            }
        }

        if ($returnAsTable)
        {
            $table = '<table>';
            $table .= '<tr><th colspan="2"><h5>Content type: ' . $filterCategoryName . '</h5></th></tr>';
            $table .= '<tr><th>Category</th><th>Param</th><th>Value</th></tr>';
            foreach ($filterArray as $categoryName => $categoryData)
            {
                foreach ($categoryData as $key => $value)
                {
                    if (is_array($value) && count($value) > 0)
                    {
                        $valueText = implode(', ', $value);
                    }
                    else
                    {
                        $valueText = '[empty]';
                    }
                    $table .= '<tr><td>' . $category . '</td><td>' . $key . '</td><td>' . $valueText . '</td></tr>';
                }
                $table .= '</table>';
                return $table;
            }
        }
        else
        {
            return $filterArray ?? [];
        }
    }


    public function parseArrayRecursive($arrayName, $array, $keyPrefix, $ignorePrefixArray = [], $replacePrefixArray = [], $debug = false)
    {
        if (isset($array) && is_array($array) && count($array) > 0)
        {
            foreach ($array as $key => $data)
            {
                $altKey = false;
                if (in_array($key, $ignorePrefixArray))
                {
                    $fullKey = $keyPrefix;
                }
                else
                {
                    if (is_array($data) === false)
                    {
                        foreach ($replacePrefixArray as $replaceKey => $replaceValue)
                        {
                            $mask = $replaceKey;
                            $mask = str_replace('*', '(.+)', $mask);
                            $mask = '/^' . $mask . '$/';
                            $debug ? $this->process->debugMessage[] = $mask : true;
                            $debug ? $this->process->debugMessage[] = $keyPrefix : true;
                            $debug ? $this->process->debugMessage[] = preg_match($mask, $keyPrefix) : true;
                            $debug ? $this->process->debugMessage[] = $key : true;
                            $debug ? $this->process->debugMessage[] = $replaceValue : true;
                            $debug ? $this->process->debugMessage[] = $data : true;
                            if (preg_match($mask, $keyPrefix) == 1 && $replaceValue == $key)
                            {
                                $keyPrefix = $data;
                                $altKey = true;
                            }
                        }
                    }
                    $fullKey = empty($keyPrefix) ? $key : $keyPrefix . '-' . $key;
                }
                if (is_array($data))
                {
                    $this->parseArrayRecursive($arrayName, $data, $fullKey, $ignorePrefixArray, $replacePrefixArray);
                }
                else
                {
                    if ($altKey === false)
                    {
                        $this->tempArray[$arrayName][$fullKey] = $data;
                    }
                }
            }
        }
    }

    public function findEstateByTextInArray($text)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select * from rko_estate_list where estate_array like :text");
            empty($text) ? $stmt->bindValue(':text', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':text', '%' . $text . '%');
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        return isset($all) ? $all[0]['estate_id'] : null;
    }

    public function getEstatePhotos($estateId)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select * from rko_photo_list where estate_id = :estate_id");
            empty($estateId) ? $stmt->bindValue(':estate_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':estate_id', $estateId);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        if (isset($all) && count($all) > 0)
        {
            foreach ($all as $row)
            {
                $finalArray[$row['file_data_id']] = $row;
            }
        }
        return isset($finalArray) ? $finalArray : null;
    }

    public function cleanEstatePhotos($estateId)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("update rko_photo_list set to_update = -1 where estate_id = :estate_id");
            empty($estateId) ? $stmt->bindValue(':estate_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':estate_id', $estateId);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        return true;
    }

    public function updatePhotoList($estateId, $fileDataId, $toUpdate, $order, $path = null, $fileName = null, $fileUpdate = null)
    {
        try
        {
            $db = $this->process->connectDb();

            if (isset($path) && isset($fileName) && isset($fileUpdate))
            {

                $stmt = $db->prepare("update rko_photo_list set photo_order = :photo_order, photo_update = :photo_update, file_path = :file_path, file_name = :file_name, to_update = :to_update where estate_id = :estate_id and file_data_id = :file_data_id");
                empty($estateId) ? $stmt->bindValue(':estate_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':estate_id', $estateId);
                empty($fileDataId) ? $stmt->bindValue(':file_data_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':file_data_id', $fileDataId);
                empty($toUpdate) ? $stmt->bindValue(':to_update', 0) : $stmt->bindValue(':to_update', $toUpdate);
                empty($order) ? $stmt->bindValue(':photo_order', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':photo_order', $order);
                empty($path) ? $stmt->bindValue(':file_path', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':file_path', $path);
                empty($fileName) ? $stmt->bindValue(':file_name', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':file_name', $fileName);
                empty($fileUpdate) ? $stmt->bindValue(':photo_update', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':photo_update', $fileUpdate);
                $stmt->execute();
                $db = null;
            }
            else
            {
                $stmt = $db->prepare("update rko_photo_list set photo_order = :photo_order, to_update = :to_update where estate_id = :estate_id and file_data_id = :file_data_id");
                empty($estateId) ? $stmt->bindValue(':estate_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':estate_id', $estateId);
                empty($fileDataId) ? $stmt->bindValue(':file_data_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':file_data_id', $fileDataId);
                empty($toUpdate) ? $stmt->bindValue(':to_update', 0) : $stmt->bindValue(':to_update', $toUpdate);
                empty($order) ? $stmt->bindValue(':photo_order', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':photo_order', $order);
                $stmt->execute();
                $db = null;

            }
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        return true;
    }

    public function insertPhotoList($estateId, $fileDataId, $toUpdate, $order, $path, $fileName, $fileUpdate)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("insert into rko_photo_list (estate_id, file_data_id, photo_order, photo_update, file_path, file_name, to_update) values (:estate_id, :file_data_id, :photo_order, :photo_update, :file_path, :file_name, :to_update)");
            empty($estateId) ? $stmt->bindValue(':estate_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':estate_id', $estateId);
            empty($fileDataId) ? $stmt->bindValue(':file_data_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':file_data_id', $fileDataId);
            empty($toUpdate) ? $stmt->bindValue(':to_update', 0) : $stmt->bindValue(':to_update', $toUpdate);
            empty($order) ? $stmt->bindValue(':photo_order', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':photo_order', $order);
            empty($path) ? $stmt->bindValue(':file_path', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':file_path', $path);
            empty($fileName) ? $stmt->bindValue(':file_name', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':file_name', $fileName);
            empty($fileUpdate) ? $stmt->bindValue(':photo_update', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':photo_update', $fileUpdate);
            $stmt->execute();
            $db = null;
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        return true;
    }

    public function imageThumbail($file, $w, $h, $crop = FALSE)
    {
        list($width, $height) = getimagesize($file);
        $r = $width / $height;
        if ($crop)
        {
            if ($width > $height)
            {
                $width = ceil($width - ($width * abs($r - $w / $h)));
            }
            else
            {
                $height = ceil($height - ($height * abs($r - $w / $h)));
            }
            $newwidth = $w;
            $newheight = $h;
        }
        else
        {
            if ($w / $h > $r)
            {
                $newwidth = $h * $r;
                $newheight = $h;
            }
            else
            {
                $newheight = $w / $r;
                $newwidth = $w;
            }
        }
        $src = imagecreatefromjpeg($file);
        $dst = imagecreatetruecolor($newwidth, $newheight);
        imagecopyresampled($dst, $src, 0, 0, 0, 0, $newwidth, $newheight, $width, $height);

        return $dst;
    }

    public function updateAgentList($agentId, $agentStamp, $agentFirstname, $agentLastname, $agentEmail, $agentPhone, $agentMobile, $agentDegreeBefore, $agentDegreeAfter, $agentGrade, $agentPhotoId, $path, $fileName, $agentUpdate, $agentToUpdate)
    {
        try
        {
            $db = $this->process->connectDb();
            $stmt = $db->prepare("update rko_agent_list set agent_stamp = :agent_stamp, firstname = :firstname, lastname = :lastname, email = :email, phone = :phone, mobile = :mobile, degree_before = :degree_before, degree_after = :degree_after, grade = :grade, photo_id = :photo_id, file_path = :file_path, file_name = :file_name, date_of_update = :date_of_update, to_update = :to_update where agent_id = :agent_id");
            empty($agentId) ? $stmt->bindValue(':agent_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':agent_id', $agentId);
            empty($agentStamp) ? $stmt->bindValue(':agent_stamp', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':agent_stamp', $agentStamp);
            empty($agentFirstname) ? $stmt->bindValue(':firstname', 0) : $stmt->bindValue(':firstname', $agentFirstname);
            empty($agentLastname) ? $stmt->bindValue(':lastname', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':lastname', $agentLastname);
            empty($agentEmail) ? $stmt->bindValue(':email', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':email', $agentEmail);
            empty($agentPhone) ? $stmt->bindValue(':phone', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':phone', $agentPhone);
            empty($agentMobile) ? $stmt->bindValue(':mobile', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':mobile', $agentMobile);
            empty($agentDegreeBefore) ? $stmt->bindValue(':degree_before', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':degree_before', $agentDegreeBefore);
            empty($agentDegreeAfter) ? $stmt->bindValue(':degree_after', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':degree_after', $agentDegreeAfter);
            empty($agentGrade) ? $stmt->bindValue(':grade', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':grade', $agentGrade);
            empty($agentPhotoId) ? $stmt->bindValue(':photo_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':photo_id', $agentPhotoId);
            empty($path) ? $stmt->bindValue(':file_path', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':file_path', $path);
            empty($fileName) ? $stmt->bindValue(':file_name', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':file_name', $fileName);
            empty($agentUpdate) ? $stmt->bindValue(':date_of_update', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':date_of_update', $agentUpdate);
            empty($agentToUpdate) ? $stmt->bindValue(':to_update', 0) : $stmt->bindValue(':to_update', $agentToUpdate);
            $stmt->execute();
            $db = null;
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        return true;
    }

    public function insertAgentList($agentId, $agentStamp, $agentFirstname, $agentLastname, $agentEmail, $agentPhone, $agentMobile, $agentDegreeBefore, $agentDegreeAfter, $agentGrade, $agentPhotoId, $path, $fileName, $agentUpdate, $agentToUpdate)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("insert into rko_agent_list (agent_id, agent_stamp, firstname, lastname, email, phone, mobile, degree_before, degree_after, grade, photo_id, file_path, file_name, date_of_update, to_update) values (:agent_id, :agent_stamp, :firstname, :lastname, :email, :phone, :mobile, :degree_before, :degree_after, :grade, :photo_id, :file_path, :file_name, :date_of_update, :to_update)");
            empty($agentId) ? $stmt->bindValue(':agent_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':agent_id', $agentId);
            empty($agentStamp) ? $stmt->bindValue(':agent_stamp', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':agent_stamp', $agentStamp);
            empty($agentFirstname) ? $stmt->bindValue(':firstname', 0) : $stmt->bindValue(':firstname', $agentFirstname);
            empty($agentLastname) ? $stmt->bindValue(':lastname', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':lastname', $agentLastname);
            empty($agentEmail) ? $stmt->bindValue(':email', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':email', $agentEmail);
            empty($agentPhone) ? $stmt->bindValue(':phone', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':phone', $agentPhone);
            empty($agentMobile) ? $stmt->bindValue(':mobile', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':mobile', $agentMobile);
            empty($agentDegreeBefore) ? $stmt->bindValue(':degree_before', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':degree_before', $agentDegreeBefore);
            empty($agentDegreeAfter) ? $stmt->bindValue(':degree_after', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':degree_after', $agentDegreeAfter);
            empty($agentGrade) ? $stmt->bindValue(':grade', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':grade', $agentGrade);
            empty($agentPhotoId) ? $stmt->bindValue(':photo_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':photo_id', $agentPhotoId);
            empty($path) ? $stmt->bindValue(':file_path', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':file_path', $path);
            empty($fileName) ? $stmt->bindValue(':file_name', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':file_name', $fileName);
            empty($agentUpdate) ? $stmt->bindValue(':date_of_update', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':date_of_update', $agentUpdate);
            empty($agentToUpdate) ? $stmt->bindValue(':to_update', 0) : $stmt->bindValue(':to_update', $agentToUpdate);
            $stmt->execute();
            $db = null;
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        return true;
    }

    public function getFilterSettings($filterName)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select * from rko_filter_settings where filter_type = :filterName");
            empty($filterName) ? $stmt->bindValue(':filterName', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':filterName', $filterName);
            $stmt->execute();
            $db = null;
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        if (isset($stmt))
        {
            $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
        return $all ?? [];
    }

    public function getContentSettings($contentName)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select * from rko_content_settings where content_type = :contentName");
            empty($contentName) ? $stmt->bindValue(':contentName', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':contentName', $contentName);
            $stmt->execute();
            $db = null;
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        if (isset($stmt))
        {
            $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
        return $all ?? [];
    }

    public function getEstateIndex()
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select id, estate_id, estate_item_id, estate_title, estate_type, commission_type, filter_category from rko_estate_list where to_update = 0");
            $stmt->execute();
            $db = null;
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        if (isset($stmt))
        {
            $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
            if (isset($all) && count($all) > 0)
            {
                foreach ($all as $row)
                {
                    $finalArray[$row['estate_id']] = '[' . $row['estate_item_id'] . ':' . $row['filter_category'] . '] ' . $row['estate_title'];
                }
            }
        }
        return $finalArray ?? [];
    }

    public function estateArticleUpdate($estateId)
    {
        $estateDetails = $this->getEstateRow($estateId);
        if (isset($estateDetails) && is_array($estateDetails) && count($estateDetails) > 0)
        {
            /** @var $processJoomla ProcessJoomla */

            if (!isset($processJoomla))
            {
                require_once 'processJoomla.php';
                $processJoomla = new ProcessJoomla($this->process);
            }
            $estateJson = json_decode($estateDetails['estate_array'], true);
            $categoryName = $estateJson['GetEstateResult']['Data']['TypeOfRealEstate']['StringValue'] . '-' . $estateJson['GetEstateResult']['Data']['TypeOfCommission']['StringValue'];
            if (empty($estateDetails['joomla_article_id']))
            {
                /* create article */
                $categoryDetails = $processJoomla->getJoomlaCategoryByTitle($categoryName);
                if (is_array($categoryDetails))
                {
                    $categoryId = $categoryDetails['id'];
                }
                else
                {
                    /* create category */
                    $categoryId = $processJoomla->createCategory($categoryName);
                }
                $articleId = $processJoomla->createArticle($categoryId, $estateId, $estateId);
                $this->updateEstateJoomlaId($estateId, $categoryId, $articleId);
            }
            else
            {
                $articleId = $estateDetails['joomla_article_id'];
                $categoryId = $estateDetails['joomla_category_id'];
            }

            /* update article content */
            $articleDetailsArray = $this->parseEstateContentToArray($estateId, false);
            $this->process->debugMessage['estateContantArray'] = $articleDetailsArray;
            $articleDetailsHtml = $this->parseEstateContentToArray($estateId, true);

            $title = $articleDetailsArray['Název nabídky'][0];
            $alias = 'estate_' . $estateId;
            $htmlIntro = '';
            $htmlFull = $this->renderEstateHtmlContent($estateId);
            $state = 1;

            $photos = $articleDetailsArray['Foto'];
            if (isset($photos) && is_array($photos) && count($photos) > 0)
            {
                $estatePhotos = $this->getEstatePhotos($estateId);
                foreach ($photos as $photoId)
                {
                    if (isset($estatePhotos) && isset($estatePhotos[$photoId]))
                    {
                        $parsedPhotos[$estatePhotos[$photoId]['photo_order']] = base64_encode($estatePhotos[$photoId]['file_path'] . $estatePhotos[$photoId]['file_name']);
                        $parsedThumbs[$estatePhotos[$photoId]['photo_order']] = base64_encode($estatePhotos[$photoId]['file_path'] . 'thumb_' . $estatePhotos[$photoId]['file_name']);
                        $parsedPhotoKeys[] = $estatePhotos[$photoId]['photo_order'];
                    }
                }
                sort($parsedPhotoKeys, SORT_NUMERIC);
                $firstImage = '?x=' . $parsedPhotos[$parsedPhotoKeys[0]];
                $firstThumb = '?x=' . $parsedThumbs[$parsedPhotoKeys[0]];
            }
            else
            {
                $firstImage = '';
                $firstThumb = '';
            }

            $images = ['float_fulltext' => "none", 'float_intro' => "none", 'image_fulltext' => "", 'image_fulltext_alt' => "", 'image_fulltext_caption' => "", 'image_intro' => "/image.php" . $firstThumb, 'image_intro_alt' => "", 'image_intro_caption' => ""];

            $updateResult = $processJoomla->updateArticle($articleId, $title, $alias, $htmlIntro, $htmlFull, $state, '*', $images);
            if ($updateResult)
            {
                $this->updateEstateJoomlaStamp($estateId);
            }
            return $updateResult;
        }
        else
        {
            return false;
        }
    }

    public function updateEstateJoomlaId($estateId, $categoryId, $articleId)
    {
        try
        {
            $db = $this->process->connectDb();
            $stmt = $db->prepare("update rko_estate_list set joomla_category_id = :categoryId, joomla_article_id = :articleId where estate_id = :estateId");
            empty($estateId) ? $stmt->bindValue(':estateId', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':estateId', $estateId);
            empty($categoryId) ? $stmt->bindValue(':categoryId', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':categoryId', $categoryId);
            empty($articleId) ? $stmt->bindValue(':articleId', 0) : $stmt->bindValue(':articleId', $articleId);
            $stmt->execute();
            $db = null;
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        return true;
    }

    public function updateEstateJoomlaStamp($estateId)
    {
        try
        {
            $db = $this->process->connectDb();
            $stmt = $db->prepare("update rko_estate_list set joomla_updated = :updated where estate_id = :estateId");
            empty($estateId) ? $stmt->bindValue(':estateId', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':estateId', $estateId);
            $stmt->bindValue(':updated', date('Y-m-d H:i:s'));
            $stmt->execute();
            $db = null;
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        return true;
    }

    public function addEstateIndex($estateId, $filterType, $filterCategory, $paramName, $valueText, $valueNum, $valueIf, $filterOrder)
    {
        try
        {
            $db = $this->process->connectDb();
            $stmt = $db->prepare("insert into rko_filter_index (estate_id, filter_type, filter_category, param_name, value_text, value_num, value_if, filter_order) values (:estate_id, :filter_type, :filter_category, :param_name, :value_text, :value_num, :value_if, :filter_order)");
            empty($estateId) ? $stmt->bindValue(':estate_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':estate_id', $estateId);
            empty($filterType) ? $stmt->bindValue(':filter_type', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':filter_type', $filterType);
            empty($filterCategory) ? $stmt->bindValue(':filter_category', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':filter_category', $filterCategory);
            empty($paramName) ? $stmt->bindValue(':param_name', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':param_name', $paramName);
            empty($valueText) ? $stmt->bindValue(':value_text', '') : $stmt->bindValue(':value_text', $valueText);
            empty($valueNum) ? $stmt->bindValue(':value_num', 0) : $stmt->bindValue(':value_num', $valueNum, PDO::PARAM_STR);
            empty($valueIf) ? $stmt->bindValue(':value_if', 0) : $stmt->bindValue(':value_if', $valueIf);
            empty($filterOrder) ? $stmt->bindValue(':filter_order', 1000000) : $stmt->bindValue(':filter_order', $filterOrder);
            $stmt->execute();
            $db = null;
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        return true;
    }

    public function addEstateIntro($estateId, $cssClass, $titleHtml, $introHtml, $firstImage, $firstThumb)
    {
        try
        {
            $db = $this->process->connectDb();
            $stmt = $db->prepare("insert into rko_intro_index (estate_id,  css_class, title_html, intro_html, first_image_link, first_thumb_link) values (:estate_id, :css_class, :title_html, :intro_html, :first_image_link, :first_thumb_link)");
            empty($estateId) ? $stmt->bindValue(':estate_id', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':estate_id', $estateId);
            empty($cssClass) ? $stmt->bindValue(':css_class', '') : $stmt->bindValue(':css_class', $cssClass);
            empty($titleHtml) ? $stmt->bindValue(':title_html', '') : $stmt->bindValue(':title_html', $titleHtml);
            empty($introHtml) ? $stmt->bindValue(':intro_html', '') : $stmt->bindValue(':intro_html', $introHtml);
            empty($firstImage) ? $stmt->bindValue(':first_image_link', '') : $stmt->bindValue(':first_image_link', $firstImage);
            empty($firstThumb) ? $stmt->bindValue(':first_thumb_link', '') : $stmt->bindValue(':first_thumb_link', $firstThumb);
            $stmt->execute();
            $db = null;
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        return true;
    }

    public function getLocality()
    { 
	try
            {
                $db = $this->process->connectDb();
           
              $query = "select filter_category,param_name,value_text,value_min,value_max, count(distinct estate_id) as pocet,filter_order from rko_filter_index_index where filter_category = 'Lokalita'  group by filter_category,param_name,value_text order by filter_order, filter_category, param_name, value_text";
                $stmt = $db->prepare($query);
                $stmt->execute();
            }
            catch (PDOException $e)
            {
                $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
                $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
                return null;
            }
            $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
            foreach ($all as $row)
            {
                $temp = null;
                
                if ( floatval($row['value_min']) <> 0 || floatval($row['value_max']) <> 0)
                {
                    $temp['value_min'] = $row['value_min'];
                    $temp['value_max'] = $row['value_max'];
                }else{
                    $temp['value_text'] = $row['value_text'];
                }
                $temp['pocet'] = $row['pocet'];

                $finalArray[$row['filter_category']][$row['param_name']][] = $temp;
            }
        
        return $finalArray ?? [];
	}
    public function getActualFilterAvailableValues($filterType = null, $estateIndexArray = [])
    {
        if (!empty($filterType))
        {
			
			 
            try
            {
                $db = $this->process->connectDb();
                if (isset($estateIndexArray) && is_array($estateIndexArray) && count($estateIndexArray) > 0)
                {
                    $query = "select filter_category,param_name,value_text, min(value_num) as value_min, max(value_num) as value_max, count(distinct estate_id) as pocet, min(filter_order) as filter_order from rko_filter_index where estate_id in (" . implode(', ', $estateIndexArray) . ") group by filter_category,param_name,value_text order by filter_order, filter_category, param_name, value_text";
                }
                else if (isset($filterType))
                {
			 
                    $query = "select filter_category,param_name,value_text,value_min,value_max, count(distinct estate_id) as pocet,filter_order from rko_filter_index_index where filter_type in ('" . $filterType . "') group by filter_category,param_name,value_text order by filter_order, filter_category, param_name, value_text";
				}
                $stmt = $db->prepare($query);
                $stmt->execute();
            }
            catch (PDOException $e)
            {
                $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
                $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
                return null;
            }
            $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
            foreach ($all as $row)
            {
                $temp = null;
                
                if ( floatval($row['value_min']) <> 0 || floatval($row['value_max']) <> 0)
                {
                    $temp['value_min'] = $row['value_min'];
                    $temp['value_max'] = $row['value_max'];
                }else{
                    $temp['value_text'] = $row['value_text'];
                }
                $temp['pocet'] = $row['pocet'];

                $finalArray[$row['filter_category']][$row['param_name']][] = $temp;
            }
        }
        return $finalArray ?? [];
    }

    public function getActualFilterAvailableCounts($filterType = null, $estateIndexArray = [])
    {
        if (!empty($filterType))
        {
            try
            {
                $db = $this->process->connectDb();
                
                if($estateIndexArray === false){
                    $query = "select a.filter_category, a.param_name, a.value_text, 0 as value_min, 0 as value_max, 0 as pocet from
                    (select filter_category,param_name,value_text,value_min,value_max from rko_filter_index_index where filter_type = '" . $filterType . "' group by filter_category,param_name,value_text) a ";
                }
                elseif (isset($estateIndexArray) && is_array($estateIndexArray) && count($estateIndexArray) > 0)
                {
                    $query = "select a.filter_category, a.param_name, a.value_text, ifnull(b.value_min, 0) as value_min, ifnull(b.value_max, 0) as value_max, ifnull(b.pocet,0) as pocet from
                    (select filter_category,param_name,value_text,value_min,value_max from rko_filter_index_index where filter_type = '" . $filterType . "' group by filter_category,param_name,value_text) a 
                    left join (select filter_category,param_name,value_text, min(value_num) as value_min, max(value_num) as value_max, count(distinct estate_id) as pocet from rko_filter_index where estate_id in (" . implode(', ', $estateIndexArray) . ") group by filter_category,param_name,value_text) b on a.filter_category = b.filter_category and  a.param_name = b.param_name and  a.value_text = b.value_text";
                }
                else if (isset($filterType))
                {
                    $query = "select a.filter_category, a.param_name, a.value_text, ifnull(b.value_min, 0) as value_min, ifnull(b.value_max, 0) as value_max, ifnull(b.pocet,0) as pocet from
                    (select filter_category,param_name,value_text,value_min,value_max from rko_filter_index_index where filter_type = '" . $filterType . "' group by filter_category,param_name,value_text) a 
                    left join (select filter_category,param_name,value_text, min(value_num) as value_min, max(value_num) as value_max, count(distinct estate_id) as pocet from rko_filter_index where filter_type = '" . $filterType . "' group by filter_category,param_name,value_text) b on a.filter_category = b.filter_category and  a.param_name = b.param_name and  a.value_text = b.value_text";
                }
                $stmt = $db->prepare($query);
                $stmt->execute();
            }
            catch (PDOException $e)
            {
                $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
                $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
                return null;
            }
            $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
            foreach ($all as $row)
            {
                $temp = null;
                
                if ( floatval($row['value_min']) <> 0 || floatval($row['value_max']) <> 0)
                {
                    $finalArray[$row['filter_category']][$row['param_name']][$row['value_text']]['value_min'] = $row['value_min'];
                    $finalArray[$row['filter_category']][$row['param_name']][$row['value_text']]['value_max'] = $row['value_max'];
                }
                $finalArray[$row['filter_category']][$row['param_name']][$row['value_text']]['pocet'] = $row['pocet'];
            }
        }
        return $finalArray ?? [];
    }

    public function searchEstateByFilter($filterType = null, $filterArray = [], $orderBy = null, $asc = true)
    {
 
        if(empty($orderBy)){
            $orderBySql = "";
        }elseif(trim($orderBy) == 'update_order'){
            $orderBySql = " order by happy_tip_order desc, " . trim($orderBy) . " " . ($asc ? "asc" : "desc");
        }else{
            $orderBySql = " order by " . trim($orderBy) . " " . ($asc ? "asc" : "desc");
        }


        $countToAsk = 0;
 
		if  (count($filterArray) > 0 && !isset($filterType))
        {
		 foreach ($filterArray as $filterCategory => $filterCategoryData)
            {
                foreach ($filterCategoryData as $paramName => $paramNameData)
                {
                    $countToAsk++;
                    foreach ($paramNameData as $paramValue)
                    {
				  
					if (isset($paramValue['value_min']) && isset($paramValue['value_max']))
							
                        {
                            /* range value */
  
						if($paramValue['value_max'] > $paramValue['value_min']) {
					 
                            $filterOr[] = "(   filter_category = '" . $filterCategory . "' and param_name = '" . $paramName . "' and value_num >= " . $paramValue['value_min'] . " and value_num <= " . $paramValue['value_max'] . ")";
						}else{
						     $filterOr[] = "(   filter_category = '" . $filterCategory . "' and param_name = '" . $paramName . "' and value_num >= " . $paramValue['value_min'] . ")";
						
						}

                        }
                        else
                        {
                        	   /*text value */
                       $filterOr[] = "(   filter_category = '" . $filterCategory . "' and param_name in ('" . $paramName . "') and value_text in ('" . $paramValue['value_text'] . "'))";
                        

						
					  }
                    }

                }
            }
			 $query = "select estate_id, count(estate_id) as pocet, price_order, update_order from rko_filter_index where " . implode(' or ', $filterOr) . " group by estate_id having pocet >= " . $countToAsk . $orderBySql;
    
			 
		} 
        elseif (count($filterArray) > 0 && isset($filterType))
        {
			
		//		preg_match("/in\s*\(\s*'([^']+)'/", $filterType, $matches);
		//	$filterType = $matches[1] ?? null;
			$filterType = preg_replace('/^WHERE\s+/i', '', $filterType);
		 
            /* create filter query */
            foreach ($filterArray as $filterCategory => $filterCategoryData)
            {
                foreach ($filterCategoryData as $paramName => $paramNameData)
                {
                    $countToAsk++;
                    foreach ($paramNameData as $paramValue)
                    {
			 
						//echo "<br>". "(filter_type = '" . $filterType . "' and filter_category = '" . $filterCategory . "' and param_name = '" . $paramName . "' and value_num >= " . $paramValue['value_min'] . " and value_num <= " . $paramValue['value_max'] . ")";;
                        if (isset($paramValue['value_min']) && isset($paramValue['value_max']))
                        {
                            /* range value */
                           if($paramValue['value_max'] > $paramValue['value_min']) {
						 
                     
						   $filterOr[] = "( " . $filterType . "  and filter_category = '" . $filterCategory . "' and param_name = '" . $paramName . "' and value_num >= " . $paramValue['value_min'] . " and value_num <= " . $paramValue['value_max'] . ")";
						   }else{
						   $filterOr[] = "( " . $filterType . "  and filter_category = '" . $filterCategory . "' and param_name = '" . $paramName . "' and value_num >= " . $paramValue['value_min'] . ")";
						    }
						
						}
                        else
                        {
                            /*text value */
                            $filterOr[] = "( " . $filterType . "  and filter_category = '" . $filterCategory . "' and param_name = '" . $paramName . "' and value_text = '" . $paramValue['value_text'] . "')";
                        }
                    }

                }
            }
            $query = "select estate_id, count(estate_id) as pocet, price_order, update_order from rko_filter_index where " . implode(' or ', $filterOr) . " group by estate_id having pocet >= " . $countToAsk . $orderBySql;
    
	

	} 
	
        elseif (isset($filterType))
        {   
		$query = "SELECT estate_id, COUNT(estate_id) AS pocet, price_order, update_order 
          FROM rko_filter_index 
			$filterType 
          GROUP BY estate_id " . $orderBySql; 
        }
        else
        { 
          
            $query = "select estate_id, count(estate_id) as pocet, price_order, update_order from rko_filter_index  group by estate_id" . $orderBySql;
        }

        $this->process->debugMessage['estate sql'] = $query;
 	// echo "SQL DOTAZ:<br>".$query."<br>" ; 
        try
        {
            $db = $this->process->connectDb();
            $stmt = $db->prepare($query);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
      $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
foreach ($all as $row)
{
    $indexa[] = $row; // uložíme celý řádek (včetně price_order)
     $index[] = $row['estate_id'];
}
$this->estateList = $index ?? [];
$this->estateList2 = $indexa ?? [];
return count($this->estateList) > 0 ? $this->estateList : false;

    }

    public function searchEstateCountByFilter($filterType = null, $filterArray = [])
    {
        if (count($filterArray) > 0 && isset($filterType))
        {
            /* create filter query */
            foreach ($filterArray as $filterCategory => $filterCategoryData)
            {
                foreach ($filterCategoryData as $paramName => $paramNameData)
                {
                    if (isset($paramNameData['value_min']) && isset($paramNameData['value_max']))
                    {
                        /* range value */
                        $filterOr[] = "(filter_type = '" . $filterType . "' and filter_category = '" . $filterCategory . "' and param_name = '" . $paramName . "' and value_num >= " . $paramNameData['value_min'] . " and value_num <= " . $paramNameData['value_max'] . ")";
                    }
                    else
                    {
                        /*text value */
                        $filterOr[] = "(filter_type = '" . $filterType . "' and filter_category = '" . $filterCategory . "' and param_name = '" . $paramName . "' and value_text = '" . $paramNameData['value_text'] . "')";
                    }

                }
            }
            $query  = "select estate_id, count(estate_id) as pocet from rko_filter_index where " . implode(' or ', $filterOr) . " group by estate_id having pocet = " . count($filterOr);
        }
        elseif (isset($filterType))
        {
            /* create type query */
            $query  = "select estate_id, count(estate_id) as pocet from rko_filter_index where filter_type = '" . $filterType . "'";

        }
        else
        {
            /* select all */
            $query  = "select estate_id, count(estate_id) as pocet from rko_filter_index";
        }

        try
        {
            $db = $this->process->connectDb();
            $stmt = $db->prepare($query);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $count = $stmt->rowCount();
        return $count ?? 0;
    }

    public function updateFilterCounts($filterType = null, $filterArray = [], $orderBy = 'update_order', $asc = false) {
        if (isset($filterArray) && is_array($filterArray) && count($filterArray) && isset($filterType))
        {
			echo "bez vyběru6";
            $estateIndex = $this->searchEstateByFilter($filterType, $filterArray, $orderBy, $asc);
        }
        elseif (isset($filterType))
        {
			
				echo "bez vyběru5";
            $estateIndex = $this->searchEstateByFilter($filterType, [], $orderBy, $asc);
        }
        else
        {
            $estateIndex = [];
        }
        $actualFilterCounts = $this->getActualFilterAvailableCounts($filterType, $estateIndex);

        return $actualFilterCounts ?? [];
    }

    public function renderFilter($filterType = null, $filterArray = [], $filterCategoryActive = [], $orderBy = 'update_order', $asc = false)
    {
		 
        /* initiate filter array from GET params */
        if (isset($_REQUEST['hhrfilterlocality']) || isset($_REQUEST['hhrfilterprice']) || isset($_REQUEST['hhrfilterpricemin']))
        {
			
		
		    $hhrFilterPredefined = true;
            if (isset($_REQUEST['hhrfilterlocality']))
            {
                /* parse locality filter array */
          /*   if($_REQUEST['hhrfilterlocality'] > 1 ) {  

 echo "<h3>". $_REQUEST['hhrfilterlocality']."</h3>";
		

			 list($hhrLocalityParamName, $hhrLocalityParamValue) = explode(':', urldecode($_REQUEST['hhrfilterlocality']));
                $filterArray['Lokalita'][$hhrLocalityParamName][]['value_text'] = $hhrLocalityParamValue;
			 	 
			 } */
			 if (!empty($_REQUEST['hhrfilterlocality'])) {

  //  echo "<h3>" . htmlspecialchars($_REQUEST['hhrfilterlocality']) . "</h3>";

    $pairs = explode(';', urldecode($_REQUEST['hhrfilterlocality'])); // rozdělíme na části

    foreach ($pairs as $pair) {
        list($paramName, $paramValue) = explode(':', $pair, 2); // jen první dvojtečka
        if (!empty($paramName) && !empty($paramValue)) {
            $filterArray['Lokalita'][$paramName][]['value_text'] = $paramValue;
        }
    }
}
			 
			}
            $priceFilterParamName = 'Cena (v Kč)';
            // $hhrPriceFilterMax = $this->getFilterMaxPrice($filterType, $priceFilterParamName);
            // $hhrPriceFilterMin = $this->getFilterMinPrice($filterType, $priceFilterParamName);
            if ((isset($_REQUEST['hhrfilterprice']) && !empty($_REQUEST['hhrfilterprice'])) || (isset($_REQUEST['hhrfilterpricemin']) && !empty($_REQUEST['hhrfilterpricemin'])))
            {
                /* parse price max filter array */
                $hhrPriceParamValue = str_replace(" ", "", urldecode($_REQUEST['hhrfilterprice'])); 
                $hhrPriceParamValuemin = str_replace(" ", "", urldecode($_REQUEST['hhrfilterpricemin']));
                $hhrPriceFilterObject['value_max'] = $hhrPriceParamValue;
                $hhrPriceFilterObject['value_min'] = $hhrPriceParamValuemin;
                $hhrPriceFilterObject['value_text'] = null;
                $filterArray[$priceFilterParamName][$priceFilterParamName][] = $hhrPriceFilterObject;
				 
        
		
		
            }
        }
        else
        {
            $hhrFilterPredefined = false;
        }

        if (isset($filterArray) && is_array($filterArray) && count($filterArray) && isset($filterType))
        {           
	
// echo "Vstup A<br>";
		   $estateIndex = $this->searchEstateByFilter($filterType, $filterArray, $orderBy, $asc);
        }
        elseif (isset($filterType))
        {
//echo "Vstup B<br>";				 
         	$estateIndex = $this->searchEstateByFilter($filterType, [], $orderBy, $asc);
        }
        else
        {
//echo "Vstup C<br>";		
		//$estateIndex = [];
			$estateIndex = $this->searchEstateByFilter(null, $filterArray, $orderBy, $asc);
			
			
        }
        if (isset($filterType))
        {
			preg_match("/in\s*\(\s*'([^']+)'/", $filterType, $matches);
			$filterType = $matches[1] ?? null;
			//echo "tady $filterType<br>";
            $hhrFilterPredefined ? $actualFilterValues = $this->getActualFilterAvailableValues($filterType, []) : $actualFilterValues = $this->getActualFilterAvailableValues($filterType, $estateIndex);
            $this->process->debugMessage['actualFilterValues'] = $actualFilterValues;
            if (isset($actualFilterValues) && count($actualFilterValues) > 0)
            {
                $categoryCounter = 0;
                $html = '<div class="hhrFilterCounter top"></div>';
                $html .= '<div class="hhrFilterControls">';
                $html .= '<div class="hhrFilterControlsContainer">';

                /* insert links to other categories */
                $filterCategoriesArray = $this->getHhrHomeFormCategories();
                if (isset($filterCategoriesArray) && is_array($filterCategoriesArray) && count($filterCategoriesArray) > 0)
                {
                    $html .= '<div class="hhrFilterCategory" data-category="Typ nabídky" data-position="0"><div class="hhrFilterCategoryLabel" onclick="hhrCategoryClick(this)" style="background-size: cover;">Typ nabídky</div>
                <div class="hhrFilterParams hidden">';
                    $filterCategoryPosition = 0;
                    foreach ($filterCategoriesArray as $filterCategoryLink => $filterCategoryLinkName)
                    {
                        $filterCategoryPosition++;
                        $filterCategorySelectedClass = $filterType == $filterCategoryLinkName ? 'selected' : 'forward';
                        $filterCategoryOnClick = $filterType == $filterCategoryLinkName ? '' : 'hhrFilterCategoryClick(this)';
                        $html .= '<div class="hhrFilterCategory" data-category="Typ nabídky - ' . $filterCategoryLinkName . '" data-position="' . $filterCategoryPosition . '" data-filter-type="' . $filterType . '"><div class="hhrFilterCategoryLabel ' . $filterCategorySelectedClass . '" onclick="' . $filterCategoryOnClick . '" data-filter-category-link="' . $filterCategoryLink . '">' . $filterCategoryLinkName . '</div></div>';
                    }
                    $html .= '</div></div>';
                }
                foreach ($actualFilterValues as $categoryName => $categoryData)
                {
                    $categoryActiveClass = (isset($filterArray) && isset($filterArray[$categoryName])) || array_search($categoryName, $filterCategoryActive) !== false ? ' active' : '';
                    $paramsHiddenClass = (isset($filterArray) && isset($filterArray[$categoryName])) || array_search($categoryName, $filterCategoryActive) !== false ? '' : ' hidden';
                    $html .= '<div class="hhrFilterCategory' . $categoryActiveClass . '" data-category="' . $categoryName . '" data-position="' . $categoryCounter++ . '">';
                    $html .= '<div class="hhrFilterCategoryLabel" onclick="hhrCategoryClick(this)">' . $categoryName . '</div>';
                    $html .= '<div class="hhrFilterParams' . $paramsHiddenClass . '">';
                    foreach ($categoryData as $paramName => $paramData)
                    {
                        if ($paramName != $categoryName)
                        {
                            $categoryActiveClass2 = (isset($filterArray) && isset($filterArray[$categoryName]) && isset($filterArray[$categoryName][$paramName])) || array_search($categoryName . ' - ' . $paramName, $filterCategoryActive) !== false ? ' active' : '';
                            $paramsHiddenClass2 = (isset($filterArray) && isset($filterArray[$categoryName]) && isset($filterArray[$categoryName][$paramName])) || array_search($categoryName . ' - ' . $paramName, $filterCategoryActive) !== false ? '' : ' hidden';
                            $html .= '<div class="hhrFilterCategory' . $categoryActiveClass2 . '" data-category="' . $categoryName . ' - ' . $paramName . '" data-position="' . $categoryCounter++ . '">';
                            $html .= '<div class="hhrFilterCategoryLabel" onclick="hhrCategoryClick(this)">' . $paramName . '</div>';
                            $html .= '<div class="hhrFilterParams' . $paramsHiddenClass2 . '">';
                        }

                        $htmlParamArray = [];

                        foreach ($paramData as $paramDataRow)
                        {
                            /* set paramKey for later sort */
                            $paramKey = $paramDataRow['value_text'];
                            /* special sorting rules */
                            if ($categoryName == 'Lokalita' && $paramName == 'Městská část')
                            {
                                $tempKey = null;
                                $tempKey = explode(' ', $paramDataRow['value_text']);
                                if (isset($tempKey[1]) && is_numeric($tempKey[1]) and strlen($tempKey[1]) === 1)
                                {
                                    $paramKey = $tempKey[0] . ' ' . '0' . $tempKey[1];
                                }
                            }
                            elseif ($categoryName == 'Dispozice')
                            {
                                $paramKey = str_replace('Garsoniera', '0', $paramKey);
                                $paramKey = str_replace('kk', '0', $paramKey);
                            }

                            $temphtml = '';
                            if (isset($paramDataRow['value_min']) && isset($paramDataRow['value_max']))
                            {
                                // $hhrFilterValueMax = $this->getFilterMaxPrice($filterType, $paramName);
                                // $hhrFilterValueMin = $this->getFilterMinPrice($filterType, $paramName);
                                // $paramDataRow['value_max'] = $hhrFilterValueMax;
                                // $paramDataRow['value_min'] = $hhrFilterValueMin;
                                $min = round($paramDataRow['value_min']);
                                $max = round($paramDataRow['value_max']);
                                $minText = number_format($min, 0, ',', ' ');
                                $maxText = number_format($max, 0, ',', ' ');
                                $temphtml .= '<div class="hhrFilterParam wide"> ';
                                $temphtml .= ' <div class="hhrFilterParamRange" data-min="' . $min . '" data-max="' . $max . '"data-category="' . $categoryName . '" data-param="' . $paramName . '" data-value="' . $paramDataRow['value_text'] . '"><div class="hhrSliderBox"><div class="hhrSlider"></div><span class="sliderMin" data-value="' . $min . '">' . $minText . '</span><span class="sliderMax" data-value="' . $max . '">' . $maxText . '</span></div></div>';
                                $temphtml .= '</div>';
                            }
                            else
                            {
                                $paramTextClass = isset($filterArray) && isset($filterArray[$categoryName]) && isset($filterArray[$categoryName][$paramName]) && array_search($paramDataRow['value_text'], array_column($filterArray[$categoryName][$paramName], 'value_text')) !== false ? ' active' : '';
                                $temphtml .= '<div class="hhrFilterParam">';
                                $temphtml .= '<div class="hhrFilterParamCheck' . $paramTextClass . '" onclick="hhrParamClick(this)" data-category="' . $categoryName . '" data-param="' . $paramName . '" data-value="' . $paramDataRow['value_text'] . '">' . $paramDataRow['value_text'] . '<span class="hhrPocet">' . $paramDataRow['pocet'] . '</span></div>';
                                $temphtml .= '</div>';
                            }
                            $htmlParamArray[$paramKey] = $temphtml;
                        }
                        if (isset($htmlParamArray) && count($htmlParamArray) > 0)
                        {
                            ksort($htmlParamArray);
                            $html .= implode('', $htmlParamArray);
                        }
                        if ($paramName != $categoryName)
                        {
                            $html .= '</div>';
                            $html .= '</div>';
                        }
                    }
                    $html .= '</div>';
                    $html .= '</div>';
                }
                $html .= '</div>';
                $html .= '</div>';
            }
            else
            {
                $html = '';
            }
       }
        else
        {
           // $html = 'select filter type';
			
			 
			
        } 
        return $html ?? '';
    }

    public function renderFilterEn($filterType = null, $filterArray = [], $filterCategoryActive = [], $orderBy = 'update_order', $asc = false)
    {
        /* initiate filter array from GET params */
        if(isset($_REQUEST['hhrfilterlocality']) || isset($_REQUEST['hhrfilterprice'])){
            $hhrFilterPredefined = true;
            if(isset($_REQUEST['hhrfilterlocality'])){
                /* parse locality filter array */
                list($hhrLocalityParamName, $hhrLocalityParamValue) = explode(':', urldecode($_REQUEST['hhrfilterlocality']));
                $filterArray['Lokalita'][$hhrLocalityParamName][]['value_text'] = $hhrLocalityParamValue;
            }
            $priceFilterParamName = 'Cena (v Kč)';
            // $hhrPriceFilterMax = $this->getFilterMaxPrice($filterType, $priceFilterParamName);
            // $hhrPriceFilterMin = $this->getFilterMinPrice($filterType, $priceFilterParamName);
            if (isset($_REQUEST['hhrfilterprice']) && !empty($_REQUEST['hhrfilterprice']))
            {
                /* parse price max filter array */
                $hhrPriceParamValue = str_replace(" ", "", urldecode($_REQUEST['hhrfilterprice']));
                $hhrPriceParamValuemin = str_replace(" ", "", urldecode($_REQUEST['hhrfilterpricemin']));
                $hhrPriceFilterObject['value_max'] = $hhrPriceParamValue;
                $hhrPriceFilterObject['value_min'] = $hhrPriceParamValuemin;
                $hhrPriceFilterObject['value_text'] = null;
                $filterArray[$priceFilterParamName][$priceFilterParamName][] = $hhrPriceFilterObject;
				
				echo " <br>min.".$hhrPriceFilterObject['value_min']."<br>";
            }
        }else{
            $hhrFilterPredefined = false;
        }

        if (isset($filterArray) && is_array($filterArray) && count($filterArray) && isset($filterType))
        {
				echo "bez vyběru3";
            $estateIndex = $this->searchEstateByFilter($filterType, $filterArray, $orderBy, $asc);
        }
        elseif (isset($filterType))
        {	echo "s filtrem";
            $estateIndex = $this->searchEstateByFilter($filterType, [], $orderBy, $asc);
        }
        else
        {
		
				echo "bez vyběru4";
            $estateIndex = [];
        }
        if (isset($filterType))
        {
            $hhrFilterPredefined ? $actualFilterValues = $this->getActualFilterAvailableValues($filterType, []) : $actualFilterValues = $this->getActualFilterAvailableValues($filterType, $estateIndex);
            $this->process->debugMessage['actualFilterValues'] = $actualFilterValues;
            if (isset($actualFilterValues) && count($actualFilterValues) > 0)
            {
                $categoryCounter = 0;
                $html = '<div class="hhrFilterCounter top"></div>';
                $html .= '<div class="hhrFilterControls">';
                $html .= '<div class="hhrFilterControlsContainer">';

                /* insert links to other categories */
                $filterCategoriesArray = $this->getHhrHomeFormCategories();
                if (isset($filterCategoriesArray) && is_array($filterCategoriesArray) && count($filterCategoriesArray) > 0)
                {
                    $html .= '<div class="hhrFilterCategory" data-category="Typ nabídky" data-position="0"><div class="hhrFilterCategoryLabel" onclick="hhrCategoryClickEn(this)" style="background-size: cover;">' . $this->translate('Typ nabídky') . '</div>
                <div class="hhrFilterParams hidden">';
                    $filterCategoryPosition = 0;
                    foreach ($filterCategoriesArray as $filterCategoryLink => $filterCategoryLinkName)
                    {
                        $filterCategoryPosition++;
                        $filterCategorySelectedClass = $filterType == $filterCategoryLinkName ? 'selected' : 'forward';
                        $filterCategoryOnClick = $filterType == $filterCategoryLinkName ? '' : 'hhrFilterCategoryClickEn(this)';
                        $html .= '<div class="hhrFilterCategory" data-category="Typ nabídky - ' . $filterCategoryLinkName . '" data-position="' . $filterCategoryPosition . '" data-filter-type="' . $filterType . '"><div class="hhrFilterCategoryLabel ' . $filterCategorySelectedClass . '" onclick="' . $filterCategoryOnClick . '" data-filter-category-link="' . $filterCategoryLink . '">' . $this->translate($filterCategoryLinkName) . '</div></div>';
                    }
                    $html .= '</div></div>';
                }
                foreach ($actualFilterValues as $categoryName => $categoryData)
                {
                    $categoryActiveClass = (isset($filterArray) && isset($filterArray[$categoryName])) || array_search($categoryName, $filterCategoryActive) !== false ? ' active' : '';
                    $paramsHiddenClass = (isset($filterArray) && isset($filterArray[$categoryName])) || array_search($categoryName, $filterCategoryActive) !== false ? '' : ' hidden';
                    $html .= '<div class="hhrFilterCategory' . $categoryActiveClass . '" data-category="' . $categoryName . '" data-position="' . $categoryCounter++ . '">';
                    $html .= '<div class="hhrFilterCategoryLabel" onclick="hhrCategoryClickEn(this)">' . $this->translate($categoryName) . '</div>';
                    $html .= '<div class="hhrFilterParams' . $paramsHiddenClass . '">';
                    foreach ($categoryData as $paramName => $paramData)
                    {
                        if ($paramName != $categoryName)
                        {
                            $categoryActiveClass2 = (isset($filterArray) && isset($filterArray[$categoryName]) && isset($filterArray[$categoryName][$paramName])) || array_search($categoryName . ' - ' . $paramName, $filterCategoryActive) !== false ? ' active' : '';
                            $paramsHiddenClass2 = (isset($filterArray) && isset($filterArray[$categoryName]) && isset($filterArray[$categoryName][$paramName])) || array_search($categoryName . ' - ' . $paramName, $filterCategoryActive) !== false ? '' : ' hidden';
                            $html .= '<div class="hhrFilterCategory' . $categoryActiveClass2 . '" data-category="' . $categoryName . ' - ' . $paramName . '" data-position="' . $categoryCounter++ . '">';
                            $html .= '<div class="hhrFilterCategoryLabel" onclick="hhrCategoryClickEn(this)">' . $this->translate($paramName) . '</div>';
                            $html .= '<div class="hhrFilterParams' . $paramsHiddenClass2 . '">';
                        }

                        $htmlParamArray = [];

                        foreach ($paramData as $paramDataRow)
                        {
                            /* set paramKey for later sort */
                            $paramKey = $paramDataRow['value_text'];
                            /* special sorting rules */
                            if($categoryName == 'Lokalita' && $paramName == 'Městská část'){
                                $tempKey = null;
                                $tempKey = explode(' ', $paramDataRow['value_text']);
                                if(isset($tempKey[1]) && is_numeric($tempKey[1]) and strlen($tempKey[1]) === 1){
                                    $paramKey = $tempKey[0] . ' ' . '0' . $tempKey[1];
                                }
                            }elseif($categoryName == 'Dispozice'){
                                $paramKey = str_replace('Garsoniera', '0', $paramKey);
                                $paramKey = str_replace('kk', '0', $paramKey);
                            }

                            $temphtml = '';
                            if (isset($paramDataRow['value_min']) && isset($paramDataRow['value_max']))
                            {
                                // $hhrFilterValueMax = $this->getFilterMaxPrice($filterType, $paramName);
                                // $hhrFilterValueMin = $this->getFilterMinPrice($filterType, $paramName);
                                // $paramDataRow['value_max'] = $hhrFilterValueMax;
                                // $paramDataRow['value_min'] = $hhrFilterValueMin;
                                $min = round($paramDataRow['value_min']);
                                $max = round($paramDataRow['value_max']);
                                $minText = number_format($min,0,',', ' ');
                                $maxText = number_format($max,0,',', ' ');
                                $temphtml .= '<div class="hhrFilterParam wide">';
                                $temphtml .= '<div class="hhrFilterParamRange" data-min="' . $min . '" data-max="' . $max. '"data-category="'.$categoryName.'" data-param="'.$paramName.'" data-value="'.$paramDataRow['value_text'] .'"><div class="hhrSliderBox"><div class="hhrSlider"></div><span class="sliderMin" data-value="' . $min . '">' . $this->translate($minText) . '</span><span class="sliderMax" data-value="' . $max . '">' . $this->translate($maxText) . '</span></div></div>';
                                $temphtml .= '</div>';
                            }
                            else
                            {
                                $paramTextClass = isset($filterArray) && isset($filterArray[$categoryName]) && isset($filterArray[$categoryName][$paramName]) && array_search($paramDataRow['value_text'], array_column($filterArray[$categoryName][$paramName], 'value_text')) !== false ? ' active' : '';
                                $temphtml .= '<div class="hhrFilterParam">';
                                $temphtml .= '<div class="hhrFilterParamCheck' . $paramTextClass . '" onclick="hhrParamClickEn(this)" data-category="'.$categoryName.'" data-param="'.$paramName.'" data-value="'.$paramDataRow['value_text'] .'">' . $this->translate($paramDataRow['value_text']) . '<span class="hhrPocet">' . $paramDataRow['pocet'] . '</span></div>';
                                $temphtml .= '</div>';
                            }
                            $htmlParamArray[$paramKey] = $temphtml;
                        }
                        if(isset($htmlParamArray) && count($htmlParamArray) > 0){
                            ksort($htmlParamArray);
                            $html .= implode('', $htmlParamArray);
                        }
                        if ($paramName != $categoryName)
                        {
                            $html .= '</div>';
                            $html .= '</div>';
                        }
                    }
                    $html .= '</div>';
                    $html .= '</div>';
                }
                $html .= '</div>';
                $html .= '</div>';
            }
            else
            {
                $html = '';
            }
        }
        else
        {
           // $html = 'select filter type';
        }
        return $html ?? '';
    }

    public function renderFilterHeader($filterType = null)
    {
        $filterTypes = $this->getFilterTypes();
        $html = '<div class="hhrHeaderControls">';
        foreach ($filterTypes as $key => $name)
        {
            $class = $key == $filterType ? 'active' : 'inactive';
            $html .= '<button class="btn hhr ' . $class . '" data-filter-type="' . $key . '" onclick="hhrFilterType(this)">' . $name . '</button>';
        }
        $html .= '</div>';
        return $html ?? '';
    }

    public function renderFilterHeaderEn($filterType = null)
    {
        $filterTypes = $this->getFilterTypes();
        $html = '<div class="hhrHeaderControls">';
        foreach ($filterTypes as $key => $name)
        {
            $class = $key == $filterType ? 'active' : 'inactive';
            $html .= '<button class="btn hhr ' . $class . '" data-filter-type="' . $key . '" onclick="hhrFilterType(this)">' . $this->translate($name) . '</button>';
        }
        $html .= '</div>';
        return $html ?? '';
    }

    public function renderEstateList($filterType = null, $filterArray = [])
    {
       if (isset($this->estateList) && count($this->estateList) > 0)
{
    $html = '<div class="hhrEstateList">';
    $countr = 0;
//    foreach ($this->estateList2 as $estate)
      foreach ($this->estateList as $estateId)
    {
//				$estateId =  $estate['estate_id'];
        $html .= '<div class="hhrEstateIntro toLoad" id="hhrIntro' . $estateId . '">';
        $html .= 'ID: ' . $estateId  . '<br>';
 //       $html .= 'Cena: ' . $estate['price_order']; // nebo si to naformátuj podle potřeby
        $html .= '</div>';
        $countr++;
    }
    $html .= '<div class="hhrFilterCounter bottom">Celkem zobrazeno: ' . $countr . '</div>';
    $html .= '</div>';
}
        else
        {
            $html = '<div class="hhrEmptyResult">Vašemu výběru bohužel neodpovídá žádná z našich nabídek</div>';
            $html .= '<div class="hhrEmptyResult">Automaticky (za <span class="hhrCountDownValue">0</span> s) Vám budou zobrazeny nabídky ze stejné kategorie, ve kterých můžete najít pro Vás nejvýhodnější alternativu.</div>';
            $html .= '<div class="hhrEmptyResultRefresh" onclick="hhrResetFilter(this)">Zobrazit jiné nabídky v této kategorii ihned</div>';
            $html .= '<script>hhrEmptyResultAutoRefresh(10);</script>';
        }
        return $html ?? '';
    }

    public function renderEstateListEn($filterType = null, $filterArray = [])
    {
        if (isset($this->estateList) && count($this->estateList) > 0)
        {
            $html = '<div class="hhrEstateList">';
            foreach ($this->estateList as $estateId)
            {
                $html .= '<div class="hhrEstateIntro toLoad" id="hhrIntro' . $estateId . '">';
                $html .= $estateId;
                $html .= '</div>';
            }
            $html .= '<div class="hhrFilterCounter bottom"></div>';
            $html .= '</div>';
        }
        else
        {
            $html = '<div class="hhrEmptyResult">' . $this->translate('Vašemu výběru bohužel neodpovídá žádná z našich nabídek') . '</div>';
            $html .= '<div class="hhrEmptyResult">' . $this->translate('Automaticky (za') . ' <span class="hhrCountDownValue">0</span> s) ' . $this->translate('Vám budou zobrazeny nabídky ze stejné kategorie, ve kterých můžete najít pro Vás nejvýhodnější alternativu.') . '</div>';
            $html .= '<div class="hhrEmptyResultRefresh" onclick="hhrResetFilterEn(this)">' . $this->translate('Zobrazit jiné nabídky v této kategorii ihned') . '</div>';
            $html .= '<script>hhrEmptyResultAutoRefreshEn(10);</script>';
        }
        return $html ?? '';
    }

    public function getFilterTypes()
    {
        try
        {
            $db = $this->process->connectDb();
            $query = "select * from altkey_rko_filter_types where id is not null";
            $stmt = $db->prepare($query);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        foreach ($all as $row)
        {
            $finalArray[$row['id']] = $row['name'];
        }
        return $finalArray;
    }

    public function getEstateIntro($estatesToRender)
    {
		
	 
     try
        {
            $db = $this->process->connectDb();
          /*  $query = "select   fi.value_text,fi.value_text as ft2, el.filter_category, el.commission_status_id, el.commission_status, el.to_update, ii.* 
            from rko_intro_index ii 
            left join rko_estate_list el on el.estate_id = ii.estate_id
            left join rko_filter_index fi on  fi.estate_id = ii.estate_id
            where fi.filter_category in ('Dispozice','Loaklita') and ii.estate_id in (" . implode(', ', $estatesToRender) . ")";
        */

            $query = "SELECT
    el.filter_category,
    el.commission_status_id,
    el.commission_status,
    el.to_update,
    ii.*,

    -- Dispozice
    (SELECT fi1.value_text 
     FROM rko_filter_index fi1 
     WHERE fi1.estate_id = ii.estate_id 
       AND fi1.filter_category = 'Dispozice' 
       AND fi1.param_name = 'Dispozice' 
     LIMIT 1) AS dispozice,

    -- Adresa složená z Město a Ulice
    CONCAT_WS(', ',
        (SELECT fi2.value_text 
         FROM rko_filter_index fi2 
         WHERE fi2.estate_id = ii.estate_id 
           AND fi2.filter_category = 'Lokalita' 
           AND fi2.param_name = 'Město' 
         LIMIT 1),
        (SELECT fi3.value_text 
         FROM rko_filter_index fi3 
         WHERE fi3.estate_id = ii.estate_id 
           AND fi3.filter_category = 'Lokalita' 
           AND fi3.param_name = 'Ulice' 
         LIMIT 1)
    ) AS adresa

FROM rko_intro_index ii
LEFT JOIN rko_estate_list el ON el.estate_id = ii.estate_id
WHERE ii.estate_id IN (" . implode(', ', $estatesToRender) . ")";


		   $stmt = $db->prepare($query);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return $all;
    }

    public function getEstateIntroEn($estatesToRender)
    {
        try
        {
            $db = $this->process->connectDb();
            $query = "select el.commission_status_id, el.commission_status, el.to_update, ii.* from rko_intro_index_en ii 
            left join rko_estate_list el on el.estate_id = ii.estate_id where ii.estate_id in (" . implode(', ', $estatesToRender) . ")";
            $stmt = $db->prepare($query);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__ , $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return $all;
    }
 
	
	
	public function renderEstateHtmlIntro($estatesToRender)
    {
        $estatesIntroArray = $this->getEstateIntro($estatesToRender);
        if (isset($estatesIntroArray) && count($estatesIntroArray) > 0)
        {
            foreach ($estatesIntroArray as $estateRow)
            {
                $disabled = $estateRow['css_class'] == 'disabled' ? true : false;

                if (!empty($estateRow['joomla_id']) && !$disabled)
                {
                    $link1 = '/?option=com_content&view=article&id=' . $estateRow['joomla_id'];
                    $link2 = '/' . $estateRow['joomla_alias'];
                }
                else
                {
                    $link1 = '';
                    $link2 = '';
                }
				 
                $specialStatus = $estateRow['commission_status_id'] == 2 ? false : true;
				 $html = ''; 
				$html .= '<div class="estate-wrapper" style="width: 100%; display: flex; flex-direction: column; gap:  10px;">';

$counter = 0;
$total = count($estatesIntroArray);

foreach ($estatesIntroArray as $i => $estateRow) { 
$ft = $estateRow['filter_category'] ;
if ($ft == "Byty-Prodej") {$estTitle="Prodej bytu";}
else if ($ft == "Byty-Pronájem") {$estTitle="Pronájem bytu";}
else if ($ft == "Domy-Prodej") {$estTitle="Prodej domu";}
else if ($ft == "Domy-Pronájem") {$estTitle="Pronájem domu";}
else if ($ft == "Komerční objekty-Pronájem") {$estTitle="Pronájem komerčního objektu";}
else if ($ft == "Komerční objekty-Prodej") {$estTitle="Prodej komerčního objektu";} 
else if ($ft == "Pozemky") {$estTitle="Pozemek";}

else{$estTitle="neidentifikováno";}
 
 
    // První 3 položky (vedle sebe)
    if ($counter === 0) {
        $html .= '<div class="first-row" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; justify-items: center;">';
    }

    if ($counter === 3) {
        $html .= '</div>'; //  first-row
        //  oddělovací div (celá šířka)
      $html .= '<div class="full-width-separator">
  <div class="adv1"><img src="img/dum_procento.png"></div>
  <div class="adv2"> Jako prodávající<br>neplatíte provizi </div>
  <div class="adv3">
    Peníze z prodeje můžete mít na účtu dřív, než se najde kupující. Díky tomu stihnete<br>
    například včas zarezervovat nové bydlení a zaplatit zálohu<br><br>
    <input type="submit" class="btn1" name="more" value="Zjisti více">
  </div>
</div>';
        //  grid pro zbývajících 8
        $html .= '<div class="grid-wrapper" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; justify-items: center;">';
    }


    if ($counter === 6) {
        // nejdřív box č. 7
        $html .= '<div  class="hhrEstateIntro ' . $estateRow['css_class'] . '" id="hhrIntro' . $estateRow['estate_id'] . '" data-estate-id="' . $estateRow['estate_id'] . '" data-estate-external-id="' . $estateRow['external_id'] . '" data-estate-status-id="' . $estateRow['commission_status_id'] . '" data-estate-toupdate="' . $estateRow['to_update'] . '" style="width:100%;
		  padding:0px">
		  '  ;
		  // 4 kus
			$html .= '<div class="itemwrapper">'; 
		$html .= '<img src="https://www.happyhouserentals.com/' . (empty($estateRow['first_thumb_link']) ? 'image.php' : $estateRow['first_thumb_link']) . '" onclick="pswpi(this)" data-photo-array="' . htmlentities($estateRow['photo_array']) . '" width=100%>';		
		 $html .= '<div class="itemwrapper3" >';
      
        $html .= '<div  class="itemsubwrapper1" >';
        $html .= '<a href="' . $link2 . '">';
        $html .= '<h4  style="font-size:14px;font-weight:600;">' . $estTitle. '</h4>'; // 
        $html .= '</a> ';
        $html .= '<div class="hhrIntroId">ID nabídky: ' . $estateRow['external_id'] . '</div>';
//        $html .= $specialStatus ? '<div class="hhrSpecialStatus"><i class="fal fa-lock-alt"></i>Rezervováno</div>' : '';
        $html .= '<div class="hhrSpecialStatus"><i class="fal fa-lock-alt"></i>Rezervováno</div>';
        $html .= '' . $estateRow['adresa'] . '';
        $html .='</div>';

        $html .= '<div class="itemsubwrapper2"  >';
        
        $html .= '<h4  style="font-size:14px;font-weight:600;">' . $estateRow['dispozice'] . '</h4></div>';
        $html .= '<div class="itemsubwrapper3" >';
        $html .= $estateRow['intro_html'];
        $html .= '</div>'; 
	// $html .= $estateRow['estate_id'] . $estateRow['css_class'] ;
		$html .='</div>';
		
		
	$html .='</div>';
               
		$html .= '</div>';
        // poté vložíme středový prvek
        $html .= '<div class="grid-center-insert" style="background: #21263a; width: 100%; height: calc(100% - 5px); display: flex; align-items: center; justify-content: center;">
		Středový box</div>';
        $counter++;
        continue;
    }

    // Generujeme běžné boxy
		$html .= '<div style="width: 100%;" class="hhrEstateIntro  " id="hhrIntro' . $estateRow['estate_id'] . '" data-estate-id="' . $estateRow['estate_id'] . '" data-estate-external-id="' . $estateRow['external_id'] . '" data-estate-status-id="' . $estateRow['commission_status_id'] . '" data-estate-toupdate="' . $estateRow['to_update'] . '">';
     	$html .= '<div class="itemwrapper">'; 
		$html .= '<img src="https://www.happyhouserentals.com/' . (empty($estateRow['first_thumb_link']) ? 'image.php' : $estateRow['first_thumb_link']) . '" onclick="pswpi(this)" data-photo-array="' . htmlentities($estateRow['photo_array']) . '" width=100%>';		
		
       
        $html .= '<div class="itemwrapper3" >';
		
      
        $html .= '<div  class="itemsubwrapper1" >';
        $html .= '<a href="' . $link2 . '">';
        $html .= '<h4  style="font-size:14px;font-weight:600;">' . $estTitle. '</h4>'; // 
        $html .= '</a> ';
        $html .= '<div class="hhrIntroId">ID nabídky: ' . $estateRow['external_id'] . '</div>';
//        $html .= $specialStatus ? '<div class="hhrSpecialStatus"><i class="fal fa-lock-alt"></i>Rezervováno</div>' : '';
        $html .= '<div class="hhrSpecialStatus"><i class="fal fa-lock-alt"></i>Rezervováno</div>';
        $html .= '' . $estateRow['adresa'] . '';
        $html .='</div>';

        $html .= '<div  class="itemsubwrapper2" >';
        
        $html .= '<h4  style="font-size:14px;font-weight:600;">' . $estateRow['dispozice'] . '</h4></div>';
        $html .= '<div  class="itemsubwrapper3" >';
        $html .= $estateRow['intro_html'];
        $html .= '</div>'; 
	// $html .= $estateRow['estate_id'] . $estateRow['css_class'] ;
		$html .='</div>';
	$html .='</div>';
	
	
	
	$html .= '</div>';

    $counter++;
}

// Ukončíme všechny otevřené kontejnery
if ($counter > 0 && $counter <= 3) {
    $html .= '</div>'; // ukončíme first-row pokud nebylo víc než 3
} elseif ($counter > 3) {
    $html .= '</div>'; // ukončíme grid-wrapper
}

$html .= '</div>'; // ukončíme estate-wrapper
 

			 
                $finalArray[$estateRow['estate_id']] = $html;
            }
        }
        return $finalArray ?? [];
    }
	

    public function renderEstateHtmlIntroEn($estatesToRender)
    {
        $estatesIntroArray = $this->getEstateIntroEn($estatesToRender);
        if (isset($estatesIntroArray) && count($estatesIntroArray) > 0)
        {
            foreach ($estatesIntroArray as $estateRow)
            {
                $disabled = $estateRow['css_class'] == 'disabled' ? true : false;

                if (!empty($estateRow['joomla_id']))
                {
                    $link1 = '/?option=com_content&view=article&id=' . $estateRow['joomla_id'];
                    $link2 = '/en/' . $estateRow['joomla_alias'];
                }
                else
                {
                    $link1 = '';
                    $link2 = '';
                }

                $specialStatus = $estateRow['commission_status_id'] == 2 ? false : true;

                $html = '<div class="hhrEstateIntro ' . $estateRow['css_class'] . '" id="hhrIntro' . $estateRow['estate_id'] . '" data-estate-id="' . $estateRow['estate_id'] . '" data-estate-external-id="' . $estateRow['external_id'] . '">';
                $html .= '<div class="hhrIntroImageA">';
                //$html .= $hrefStart1;
                $html .= '<div class="hhrIntroImage" style="background-image: url(/' . (empty($estateRow['first_thumb_link']) ? 'image.php' : $estateRow['first_thumb_link']) . ');"  onclick="pswpi(this)" data-photo-array="' . htmlentities($estateRow['photo_array']) . '"></div>';
                //$html .= $hrefEnd;
                $html .= '<div class="hhrIntroControls">';
                $html .= '<div class="hhrIconbutton addFavorite" title="' . $this->translate('Přidat k oblíbeným') . '" onclick="hhrAddFavorite(this,' . $estateRow['estate_id'] . ',\'' . $estateRow['external_id'] . '\')"><i class="fal fa-heart" aria-hidden="true"></i></div>';
                if ($specialStatus)
                {
                    $html .= '<div class="hhrIconbutton addPlan" title="' . $this->translate('Nabídka není aktuálně dostupná') . '" ><i class="fal fa-lock-alt" aria-hidden="true"></i></div>';
                }
                else
                {
                    $html .= '<div class="hhrIconbutton addPlan" title="' . $this->translate('Přidat do seznamu prohlídek') . '" onclick="hhrAddPlan(this,' . $estateRow['estate_id'] . ',\'' . $estateRow['external_id'] . '\')"><i class="fal fa-calendar-check" aria-hidden="true"></i></div>';
                }
                if ($disabled)
                {
                    $html .= '<a href="" rel="noopener noreferrer"  class="hhrIconbutton" title="' . $this->translate('Otevřít kontaktní formulář') . '" onclick="hhrOpenContactFormEn(this)">';
                    $html .= '<i class="fal fa-envelope" aria-hidden="true"></i>';
                    $html .= '</a>';
                }
                else
                {
                    $html .= '<a href="' . $link2 . '" target="_blank" rel="noopener noreferrer"  class="hhrIconbutton" title="' . $this->translate('Otevřít v samostatném okně') . '">';
                    $html .= '<i class="fal fa-external-link" aria-hidden="true"></i>';
                    $html .= '</a>';
                }
                $html .= '</div>';
                $html .= '</div >';
                $html .= '<div class="hhrIntroHeader">';
                if ($disabled)
                {
                    $html .= '<a href="" title="Open contact form" onclick="hhrOpenContactForm(this)">';
                    $html .= '<h4>' . $estateRow['title_html'] . '</h4>';
                    $html .= '</a>';
                    $html .= '<div class="hhrIntroId">Estate ID: ' . $estateRow['external_id'] . '</div>';
                }
                else
                {
                    $html .= '<a href="' . $link2 . '">';
                    $html .= '<h4>' . $estateRow['title_html'] . '</h4>';
                    $html .= '</a>';
                    $html .= '<div class="hhrIntroId">Estate ID: ' . $estateRow['external_id'] . '</div>';
                    $html .= $specialStatus ? '<div class="hhrSpecialStatus"><i class="fal fa-lock-alt"></i>' . $this->translate('Rezervováno') . '</div>' : '';
                }
                $html .= '</div>';
                $html .= '<div class="hhrIntroA">';
                $html .= '<div class="hhrIntro">' . $estateRow['intro_html'] . '</div>';
                $html .= '</div>';
                $html .= '</div>';
                $finalArray[$estateRow['estate_id']] = $html;
            }
        }
        return $finalArray ?? [];
    }

    public function renderNextloadButton($pageSize){
        $html = '<div class="hhrLoadNext"><button class="btn btn-success btn-hhr" onclick="hhrLoadNext(this, ' . $pageSize . ')">Zobrazit další</button></div>';
        return $html;
    }

    public function renderNextloadButtonEn($pageSize)
    {
        $html = '<div class="hhrLoadNext"><button class="btn btn-success btn-hhr" onclick="hhrLoadNextEn(this, ' . $pageSize . ')">' . $this->translate('Zobrazit další') . '</button></div>';
        return $html;
    }

    public function renderEstateHtmlContent($estateId)
    {

        $estateDetails = $this->getEstateRow($estateId);
        $estateJson = json_decode($estateDetails['estate_array'], true);

        $articleDetailsArray = $this->parseEstateContentToArray($estateId, false);

        $categoryName = $estateJson['GetEstateResult']['Data']['TypeOfRealEstate']['StringValue'];
        $commissionType = $estateJson['GetEstateResult']['Data']['TypeOfCommission']['StringValue'];

        $this->process->debugMessage['articleDetailsArray'] = $articleDetailsArray;

        $price = number_format($articleDetailsArray['Content']['Cena'][0] ?? 0, 2, ',', ' ') . ' ' . ($articleDetailsArray['Content']['Cena v měně'][0] ?? 'Kč');
        if($commissionType == 'Pronájem'){
            $price .= ' / měsíc';
        }
        $title = $articleDetailsArray['Content']['Název nabídky'][0];
        $titleHtml = $title;

        $htmlHeader = '<div class="hhrArticle hhrHeader">';
        $htmlHeader .= '<h1>' . $titleHtml . '</h1>';
        $htmlHeader .= '<div class="hhrCategory">' . $categoryName . ' - ' . $commissionType . '</div>';
        $htmlHeader .= '</div>';

        $htmlPrice = '<div class="hhrArticle hhrPrice">';
        $htmlPrice .= '<div class="hhrPriceMain">' . $price. '</div>';
        $htmlPrice .= '<div class="hhrPriceComment">' .$articleDetailsArray['Content']['Doplnění ceny'][0] . '</div>';
        $htmlPrice .= '</div>';

        $photos = $articleDetailsArray['Content']['Foto'];
        if (isset($photos) && is_array($photos) && count($photos) > 0)
        {
            $estatePhotos = $this->getEstatePhotos($estateId);
            foreach ($photos as $photoId)
            {
                if (isset($estatePhotos) && isset($estatePhotos[$photoId]))
                {
                    $parsedPhotos[$estatePhotos[$photoId]['photo_order']] = base64_encode($estatePhotos[$photoId]['file_path'] . $estatePhotos[$photoId]['file_name']);
                    $parsedThumbs[$estatePhotos[$photoId]['photo_order']] = base64_encode($estatePhotos[$photoId]['file_path'] . 'thumb_' . $estatePhotos[$photoId]['file_name']);
                    $parsedPhotoKeys[] = $estatePhotos[$photoId]['photo_order'];
                }
            }
            sort($parsedPhotoKeys, SORT_NUMERIC);
            $firstImage = '?x=' . $parsedPhotos[$parsedPhotoKeys[0]];
            $firstThumb = '?x=' . $parsedThumbs[$parsedPhotoKeys[0]];

            $htmlPhoto = '<div class="hhrArticle hhrPhotos">';
            $htmlPhoto .= '<div class="hhrPhotoMain" style="background-image: url(/image.php' . $firstImage . ');">';
            $htmlPhoto .= '</div>';
            $htmlPhoto .= '<div class="hhrThumbs">';
            $htmlPhoto .= '<ul>';
            foreach($parsedPhotoKeys as $photoKey){
                $htmlPhoto .= '<li data-photo-url="/image.php?x=' . $parsedPhotos[$photoKey] . '">';
                $htmlPhoto .= '<div class="hhrThumb" style="background-image: url(/image.php?x=' . $parsedThumbs[$photoKey] . ');">';
                $htmlPhoto .= '</div>';
                $htmlPhoto .= '</li>';
            }
            $htmlPhoto .= '</ul>';
            $htmlPhoto .= '</div>';
            $htmlPhoto .= '</div>';

        }
        else
        {
            $firstImage = '';
            $firstThumb = '';
        }

        $htmlText = '<div class="hhrArticle hhrText">';
        $htmlText .= '<p>' . $articleDetailsArray['Content']['Popis nabídky'][0] . '</p>';
        $htmlText .= '</div>';

        $paramsOrdered = $this->getParamsOrder($categoryName . '-' . $commissionType);
        foreach($paramsOrdered as $paramName){
            if(isset($articleDetailsArray['Params'][$paramName]) && is_array($articleDetailsArray['Params'][$paramName])){
                $params[$paramName] = implode(', ', $articleDetailsArray['Params'][$paramName]);
            }
        }
        if(isset($params) && count($params) > 0){
            $htmlParams = '<div class="hhrArticle hhrParams">';
            $htmlParams .= '<ul>';
            foreach($params as $paramName => $paramValue){
                $htmlParams .= '<li>';
                $htmlParams .= '<div class="hhrParamLabel">' . $paramName . ' :</div>';
                $htmlParams .= '<div class="hhrParamValue">' . $paramValue . '</div>';
                $htmlParams .= '</li>';
            }
            $htmlParams .= '</ul>';
            $htmlParams .= '</div>';
        }

        $agentId = $estateDetails['agent_id'];
        $agentDetails = $this->getAgentRow($agentId);
        if(count($agentDetails) > 0){
            $htmlAgent = '<div class="hhrArticle hhrAgent">';
            
            $agentPhoto = base64_encode($agentDetails['file_path'] . $agentDetails['file_name']);
            $htmlAgent .= '<div class="hhrPhotoAgent" style="background-image: url(/image.php?x=' . $agentPhoto . ');"></div>';

            $agentName = empty($agentDetails['degree_before']) ? '' : $agentDetails['degree_before'] . '&nbsp;';
            $agentName .= empty($agentDetails['firstname']) ? '' : $agentDetails['firstname'] . '&nbsp;';
            $agentName .= empty($agentDetails['lastname']) ? '' : $agentDetails['lastname'] . '&nbsp;';
            $agentName .= empty($agentDetails['degree_after']) ? '' : $agentDetails['degree_after'];
            $htmlAgent .= '<div class="hhrAgentName">' .  $agentName . '</div>';
            $htmlAgent .= '<div class="hhrAgentContact"><a href="mailto:' .$agentDetails['email']. '">' .  $agentDetails['email'] . '</a></div>';
            $htmlAgent .= '<div class="hhrAgentContact"><a href="callto:' .$agentDetails['mobile']. '">' .  $agentDetails['mobile'] . '</a></div>';

            $htmlAgent .= '</div>';

        }else{
            $htmlAgent = '';
        }

        $mapLat = $articleDetailsArray['Content']['GPSLat'][0];
        $mapLng = $articleDetailsArray['Content']['GPSLng'][0];

        if(!empty($mapLat) && !empty($mapLng)){
            $annonymous = true;
            $zoom = 18;
            $mapStyle = 'satellite';

            $mapSrc = 'https://www.google.com/maps/embed/v1/';
            $mapSrc .= $annonymous ? 'view' : 'place';
            $mapSrc .= '?key=AIzaSyA0qXGiwNzQpuCFDN8nLL_jrjUGk91bKX0';
            $mapSrc .= $annonymous ? '' : '&q=' . htmlspecialchars('Happy House Rentals');
            $mapSrc .= '&center=';
            $mapSrc .= $mapLat;
            $mapSrc .= ',';
            $mapSrc .= $mapLng;
            $mapSrc .= '&zoom=';
            $mapSrc .= $zoom;
            $mapSrc .= '&maptype=';
            $mapSrc .= $mapStyle;

            $htmlMap = '<div class="hhrArticle hhrMap">';
            $htmlMap .= '<iframe class="hhrMapFrame" src="' . $mapSrc .'" />';
            $htmlMap .= '</div>';
    
        }else{
            $htmlMap = '';
        }

        if(!empty($mapLat) && !empty($mapLng)){
            $annonymous = true;
            $zoom = 18;
            $mapStyle = 'satellite';

            $htmlJsMap = '<div class="hhrArticle hhrMap" id="hhrSingleMap" data-zoom="' . $zoom .'" data-style="' . $mapStyle . '" data-lat="' . $mapLat . '" data-lng="' . $mapLng . '" data-marker="' . $titleHtml . '">';
            $htmlJsMap .= '</div>';

            $htmlJsMap .= '<script async defer src="https://maps.googleapis.com/maps/api/js?key=AIzaSyA0qXGiwNzQpuCFDN8nLL_jrjUGk91bKX0&v=3.exp&callback=hhrInitMap"></script>';

        }else{
            $htmlJsMap = '';
        }

        return $htmlHeader . $htmlPrice . $htmlPhoto . $htmlText . $htmlParams . $htmlAgent . $htmlJsMap;
    }

    public function getWelcomeMessage()
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select message_cz, message_en from rko_welcome_message where (show_from is null or show_from < now()) and (show_to is null or show_to > now())");
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        return isset($all) ? $all : null;
    }

    public function getFilterMaxPrice($filterType, $paramName)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select max(value_num) as value from rko_filter_index where filter_type = :filter_type and param_name = :param_name");
            empty($filterType) ? $stmt->bindValue(':filter_type', null, PDO::PARAM_NULL) : $stmt->bindValue(':filter_type', $filterType);
            empty($paramName) ? $stmt->bindValue(':param_name', null, PDO::PARAM_NULL) : $stmt->bindValue(':param_name', $paramName);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        return isset($all) ? $all[0]['value'] : null;
    }

    public function getFilterMinPrice($filterType, $paramName)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select min(value_num) as value from rko_filter_index where filter_type = :filter_type and param_name = :param_name");
            empty($filterType) ? $stmt->bindValue(':filter_type', null, PDO::PARAM_NULL) : $stmt->bindValue(':filter_type', $filterType);
            empty($paramName) ? $stmt->bindValue(':param_name', null, PDO::PARAM_NULL) : $stmt->bindValue(':param_name', $paramName);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        return isset($all) ? $all[0]['value'] : null;
    }

    public function getJoomlaAliasByEstateId($estateId)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select ii.joomla_alias from rko_intro_index ii join rko_estate_list el on el.estate_item_id = :estateId and el.estate_id = ii.estate_id where ii.css_class != 'disabled'");
            empty($estateId) ? $stmt->bindValue(':estateId', '000') : $stmt->bindValue(':estateId', $estateId);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        return isset($all) ? $all[0]['joomla_alias'] : null;
    }

    public function getJoomlaAliasByEstateIdEn($estateId)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select ii.joomla_alias from rko_intro_index_en ii join rko_estate_list el on el.estate_item_id = :estateId and el.estate_id = ii.estate_id where ii.css_class != 'disabled'");
            empty($estateId) ? $stmt->bindValue(':estateId', '000') : $stmt->bindValue(':estateId', $estateId);
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        return isset($all) ? $all[0]['joomla_alias'] : null;
    }

    public function getHhrHomeFormCategories()
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select count(el.estate_id) as pocet, el.filter_category, cl.link from rko_estate_list el left join rko_category_link cl on cl.category = el.filter_category where  el.joomla_article_id is not null and el.to_update = 0 and el.commission_status_id = 2  group by filter_category having pocet > 0");
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        if (isset($all) && is_array($all))
        {
            foreach ($all as $row)
            {
                $resultArray[$row['link']] = $row['filter_category'];
            }
        }
        return isset($resultArray) ? $resultArray : null;

    }

    public function getHhrHomeFormCategoriesEn()
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select count(el.estate_id) as pocet, el.filter_category, cl.link_en as link from rko_estate_list el left join rko_category_link cl on cl.category = el.filter_category where  el.joomla_article_id is not null and el.to_update = 0 and el.commission_status_id = 2  group by filter_category having pocet > 0");
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        if (isset($all) && is_array($all))
        {
            foreach ($all as $row)
            {
                $resultArray[$row['link']] = $this->translate($row['filter_category']);
            }
        }
        return isset($resultArray) ? $resultArray : null;

    }

    public function getHhrHomeFormLocationsEn()
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select distinct concat(param_name, ':' , value_text) as lokalita, param_name, value_text  from rko_filter_index where filter_category = 'Lokalita' and param_name != 'Čtvrť'  and param_name != 'Kraj' order by lokalita");
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        if (isset($all) && is_array($all))
        {
            foreach ($all as $row)
            {
                $resultArray[$row['lokalita']] = $this->translate($row['param_name']) . ':' . $row['value_text'];
            }
        }
        return isset($resultArray) ? $resultArray : null;

    }
	
	
 
 public function getEstateAddresses()
{
    try {
        $db = $this->process->connectDb();

        $stmt = $db->prepare("
            SELECT estate_id, param_name, value_text  
            FROM rko_filter_index  
            WHERE filter_category = 'Lokalita'  
        ");
        $stmt->execute();
    } catch (PDOException $e) {
        return [];
    }

    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $db = null;

    $addresses = [];

    foreach ($rows as $row) {
        $id = $row['estate_id'];
        $param = $row['param_name'];
        $value = $row['value_text'];

        if (!isset($addresses[$id])) {
            $addresses[$id] = [];
        }

        $addresses[$id][$param] = $value;
    }

    $result = [];
    $seen = []; // pro sledování unikátních záznamů

    foreach ($addresses as $parts) {
        if (empty($parts['Ulice'])) {
            continue;
        }

        $zobrazit = [];
        $retezec = [];

        foreach (['Město', 'Městská část', 'Čtvrť', 'Ulice'] as $p) {
            if (!empty($parts[$p])) {
                $zobrazit[] = $parts[$p];
                $retezec[] = $p . ':' . $parts[$p];
            }
        }

        if (!empty($zobrazit)) {
            $label = implode(', ', $zobrazit);
            $data = implode(';', $retezec);

            // Pokud už stejný záznam byl přidán, přeskočíme
            $hash = md5($label . $data);
            if (isset($seen[$hash])) {
                continue;
            }
            $seen[$hash] = true;

            $result[] = [
                'label' => $label,
                'value' => $label,
                'data'  => $data
            ];
        }
    }

    return $result;
}




    public function getHhrHomeFormLocations()
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select distinct concat(param_name, ':' , value_text) as lokalita from rko_filter_index where filter_category = 'Lokalita' and param_name != 'Čtvrť'  and param_name != 'Kraj' order by lokalita");
            $stmt->execute();
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $db = null;
        if (isset($all) && is_array($all))
        {
            foreach ($all as $row)
            {
                $resultArray[$row['lokalita']] = $row['lokalita'];
            }
        }
        return isset($resultArray) ? $resultArray : null;

    }

    public function getEstateAgentsListByEstateIdArray($hhrPlanList){

        $keys = isset($hhrPlanList) && is_array($hhrPlanList) ? array_keys($hhrPlanList) : null;

        if (isset($keys) && is_array($keys))
        {
            try
            {
                $db = $this->process->connectDb();

                $stmt = $db->prepare("select distinct a.firstname, a.lastname, a.degree_before, a.degree_after, a.email from rko_agent_list a left join rko_estate_list e on e.agent_id = a.agent_id where e.estate_id in (" . implode(',', $keys) . ")");
                $stmt->execute();
            }
            catch (PDOException $e)
            {
                $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
                $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
                return null;
            }
            $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $db = null;
            if (isset($all) && is_array($all))
            {
                foreach ($all as $row)
                {
                    $resultArray[$row['email']]['degree_before'] = $row['degree_before'];
                    $resultArray[$row['email']]['degree_after'] = $row['degree_after'];
                    $resultArray[$row['email']]['firstname'] = $row['firstname'];
                    $resultArray[$row['email']]['lastname'] = $row['lastname'];
                }
            }
        }
        return isset($resultArray) ? $resultArray : null;
    }

    public function getHhrEstateStatus($estateExternalId)
    {
        try
        {
            $db = $this->process->connectDb();

            $stmt = $db->prepare("select l.commission_status_id, l.commission_status, l.to_update, ii.css_class, c.link, c.link_en from rko_estate_list l left join rko_category_link c on c.category = l.filter_category left join rko_intro_index ii on ii.external_id = l.estate_item_id where l.estate_item_id = :estateId order by l.id desc");
            empty($estateExternalId) ? $stmt->bindValue(':estateId', NULL, PDO::PARAM_NULL) : $stmt->bindValue(':estateId', $estateExternalId);
            $stmt->execute();
            $db = null;
        }
        catch (PDOException $e)
        {
            $this->process->sendEmailError($this->process->conf->emailAdmin, $this->process->conf->appName . " System error " . __FUNCTION__, $e->getMessage(), $this->process->mailHeader);
            $this->process->debugMessage[] = $this->process->conf->appName . " System error " . __FUNCTION__ . PHP_EOL . $e->getMessage();
            return null;
        }
        if (isset($stmt))
        {
            $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
        return isset($all) && is_array($all) && isset($all[0]) ? $all[0] : [];
    }

    public function renderFinalFilterHtml($filterType = null){
        if (!empty($filterType))
        {
            //$header = '<div class="hhrHeader">' . $this->renderFilterHeader($filterType) . '</div>';
            $filter = '<div class="hhrFilter">' . $this->renderFilter($filterType) . '</div>';
            $content = '<div class="hhrContent">' . $this->renderEstateList($filterType) . '</div>';
        }
        else
        {
            //$header = '<div class="hhrHeader">' . $this->renderFilterHeader() . '</div>';
            $filter = '<div class="hhrFilter">' . $this->renderFilter() . '</div>';
            $content = '<div class="hhrContent">' . $this->renderEstateList() . '</div>';
        }
        $page = '<div class="hhrPage">' . $filter . $content . '</div>';
        return $page;
    }

    public function renderFinalFilterHtmlEn($filterType = null)
    {
        if (!empty($filterType))
        {
            //$header = '<div class="hhrHeader">' . $this->renderFilterHeaderEn($filterType) . '</div>';
            $filter = '<div class="hhrFilter">' . $this->renderFilterEn($filterType) . '</div>';
            $content = '<div class="hhrContent">' . $this->renderEstateListEn($filterType) . '</div>';
        }
        else
        {
            //$header = '<div class="hhrHeader">' . $this->renderFilterHeaderEn() . '</div>';
            $filter = '<div class="hhrFilter">' . $this->renderFilterEn() . '</div>';
            $content = '<div class="hhrContent">' . $this->renderEstateListEn() . '</div>';
        }
        $page = '<div class="hhrPage">' . $filter . $content . '</div>';
        return $page;
    }

}

/**
 * Process Module
 * Handles communication RKO database
 */

 class ProcessMini
 {
      /**
     * Initialize PDO connection with primary database 
     *
     * @return object PDO database link
     */

     public $emailHeader;
     public $mailHeader;
     public $logged;
     public $conf;
    public $debugMessage;

    function __construct()
    {
        $this->conf = new stdClass();
        /*
		$this->conf->dtbHost = 'localhost';
        $this->conf->dtbName = '24743_newhhr2';
        $this->conf->dtbUser = '24743_44834';
        $this->conf->dtbPassword = 'newhhr2020';
*/
		$this->conf->dtbHost = 'localhost';
        $this->conf->dtbName = 'happyhouserentals';
        $this->conf->dtbUser = 'czechbuh';
        $this->conf->dtbPassword = 'Pektusan2744';
        $this->conf->emailFrom = 'info@happyhouserentals.com';
        $this->conf->emailAdmin = 'czechbuh@seznam.cz';
		
		
        $this->conf->appName = 'HHR-NEW';

        $this->mailHeader = 'From: ' . $this->conf->appName . ' <' . $this->conf->emailFrom . '>' . "\r\n" . 'Reply-To: ' . $this->conf->appName . ' <' . $this->conf->emailFrom . '>' . "\r\n" . 'X-Mailer: NasSystem';

        $this->logged = true;
    }

    public function connectDb()
    {
        $db = new \PDO('mysql:host=' . $this->conf->dtbHost . ';dbname=' . $this->conf->dtbName . ';charset=utf8mb4', $this->conf->dtbUser, $this->conf->dtbPassword);
        $db->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
        $db->setAttribute(\PDO::ATTR_EMULATE_PREPARES, true);
        $userName = 'NEW';
        $teamId = 'WEB';
        $actualIp = $_SERVER['REMOTE_ADDR'];
        $db->exec("SET sql_mode=(SELECT REPLACE(@@sql_mode,'ONLY_FULL_GROUP_BY',''))");
        $db->exec("SET @tempUser = '" . $userName . " [team:" . $teamId . "] (ip:" . $actualIp . ")'");

        return $db;
    }

    public function sendEmailError($emailAdmin, $errorSubject, $errorMessage, $emailHeaders = null)
    {
        return true;
    }

}


/**
 * Email Module
 * Handles email requests
 */

 class ProcessMail
 {
      /**
     * Initialize PHP mail class 
     *
     * @return object PDO database link
     */

    /* MAIL SETTINGS */
    public $appName = 'HHR WEB portal';
    public $blockEmails = false;
    public $blockErrorEmails = true;
    public $isSMTP = true;
    public $smtpServer = 'smtp.office365.com';
    public $smtpUser = 'cc_smtp@happyhouserentals.com';
    public $smtpPassword = '.XwckNRWu-7';
    public $smtpPort = 587;
    public $smtpSSL = 'tls';
    public $imapServer = 'imap.ebola.cz';
    public $imapUser = 'rko@hk-press.cz';
    public $imapPassword = '65hkpress65';
    public $imapPort = 993;
    public $imapSSL = 'SSL';
    public $emailFrom = 'info@happyhouserentals.com';
    public $emailNotificationFrom = 'info@happyhouserentals.com';
    public $emailReplyTo = 'info@happyhouserentals.com';
    public $emailSupport = 'support@hk-press.cz';
    public $emailAdmin = 'support@hk-press.cz';
    // public $emailHhrAdmin = 'mikanova@happyhouserentals.com';
    public $emailHhrAdmin = 'support@hk-press.cz';
    public $mailer = '/data/www/24743/happyhouserentals_com/rko/secret/libs/PHPmailer/PHPMailerAutoload.php';
    public $emailLog = '/data/www/24743/happyhouserentals_com/rko/secret/log/email-web-XXX.log';
    public $emailSubjectPrefix = 'HHR Web Email';
    public $emailFooter = 'This email was sent from HHR Web Portal, and the portal is ready to recieve your answer even with attachments, so please feel free to simply reply to this message';
    public $emailSeparator = '------------ please do not change content below ------------';

    public $phpMail;

    function __construct()
    {
        require_once $this->mailer;
        //Create a new PHPMailer instance
        $this->phpMail = new PHPMailer;

        if($this->isSMTP){
            // set SMTP params
            $this->phpMail->IsSMTP(); // telling the class to use SMTP
            $this->phpMail->SMTPDebug = 0; // enables SMTP debug information (for testing) // 1 = errors and messages // 2 = messages only
            $this->phpMail->SMTPAuth = true; // enable SMTP authentication
            if (isset($this->smtpSSL))
            {
                $this->phpMail->SMTPSecure = "tls"; // sets the prefix to the servier
                $this->phpMail->SMTPAutoTLS = true;
            }
            else
            {
                $this->phpMail->SMTPAutoTLS = true;
            }
            $this->phpMail->Host = $this->smtpServer; // sets the SMTP server
            $this->phpMail->Port = isset($this->smtpPort) ? $this->smtpPort : 465; // set the SMTP port
            $this->phpMail->Username = $this->smtpUser; // SMTP account username
            $this->phpMail->Password = $this->smtpPassword; // SMTP account password
            $this->phpMail->SMTPOptions = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
        }
        $this->phpMail->CharSet = 'UTF-8';

        $this->emailLog = str_replace('XXX', date('Ymd'), $this->emailLog);
    }

    public function sendHhrForm($receiver, $sender, $subject, $message, $formValues)
    {

        $this->phpMail->setFrom($this->emailNotificationFrom, $this->appName . ' Admin');
        $this->phpMail->addAddress($receiver);
        $this->phpMail->addReplyTo($sender);
        $this->phpMail->addCC($sender);

        if ($this->emailAdmin !== $receiver)
        {
            $this->phpMail->addBCC($this->emailAdmin);
        }

        if ($this->emailHhrAdmin !== $receiver && $this->emailHhrAdmin !== $sender)
        {
            $this->phpMail->addBCC($this->emailHhrAdmin);
        }

        $this->phpMail->Subject = $subject;
        $messageHtml = $message ;

        /*
        if (isset($formValues) && is_array($formValues))
        {
            foreach ($formValues as $formItem)
            {
                $itemName = $formItem['name'] ?? 'value';
                $itemValue = $formItem['value'] ?? 'empty';
                $messageHtml .= '<br>';
                $messageHtml .= $itemName . ': ' . $itemValue;
            }
        }

        $messageHtml .= '<br><br>Your Message Robot';
        */

        $this->phpMail->msgHTML($messageHtml);

        //send the message, check for errors
        $res = $this->blockEmails ? true : $this->phpMail->send();
        if (!$res)
        {
            $phpMailerResult = "Mailer Error: " . $this->phpMail->ErrorInfo;
            $result = false;
        }
        else
        {
            $phpMailerResult = $this->blockEmails ? "Message blocked" : "Message sent!";
            $result = true;
        }

        file_put_contents($this->emailLog, PHP_EOL . date('Y-m-d H:i:s') . ' ********** Sent mail via PHPmailer **********' . PHP_EOL, FILE_APPEND);
        file_put_contents($this->emailLog, 'PHPmailer RESULT:' . PHP_EOL . $phpMailerResult . PHP_EOL . PHP_EOL, FILE_APPEND);
        file_put_contents($this->emailLog, 'Email subject:' . PHP_EOL . $this->phpMail->Subject . PHP_EOL . PHP_EOL, FILE_APPEND);
        $finalRecipients = $this->phpMail->getAllRecipientAddresses();
        file_put_contents($this->emailLog, 'Email recipients:' . PHP_EOL . json_encode($finalRecipients) . PHP_EOL . PHP_EOL, FILE_APPEND);
        file_put_contents($this->emailLog, 'PHPmailer RESULT:' . PHP_EOL . $phpMailerResult . PHP_EOL . PHP_EOL, FILE_APPEND);
        file_put_contents($this->emailLog, '*****************************' . PHP_EOL, FILE_APPEND);
        return $result;
    }

}

